/**************************************************************************************************
  Filename:       sample_main_rc.cpp

  Description:    SimpleConsole application (RC node)

  Copyright 2009 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#include "rti.h"

// Pairing reference
uint8 destIdx;

// Character read from console
int ch;

//cache for data to be transmitted
uint8 data[2];

// check that a valid key is pressed
int validKey = 0;

int main(int argc, char **argv)
{
  // Pass one argument as COM port path
  if (argc < 2)
  {
    fprintf(stderr, "Usage: SampleConsole <com_port_path>\n");
    exit(-1);
  }

  // Initialize RTI windows module, pass COM port path as argument
  if (!RTI_InitWin32Module(argv[1]))
  {
    fprintf(stderr, "Failed to start RTI library module\n");
    exit(-1);
  }

 
  // Configure node capabilities
  // node capabilities: controller, battery-powered, security-capable, ch-norm-incapable
  uint8 nodeCap = RTI_BUILD_NODE_CAPABILITIES(0, 0, 1, 0);
  if (RTI_SUCCESS != RTI_WriteItem(RTI_CP_ITEM_NODE_CAPABILITIES, 1, &nodeCap))
  {
    RTI_CloseWin32Module();
    fprintf(stderr, "Failed to write node capabilities. Please check the connection\n");
    exit(-1);
  }

  // Configure node device type
  uint8 devType = RTI_DEVICE_REMOTE_CONTROL;
  RTI_WriteItem(RTI_CP_ITEM_APPL_DEV_TYPE_LIST, 1, &devType);

  // Configure node supported target device types
  devType = RTI_DEVICE_TELEVISION;
  RTI_WriteItem(RTI_CP_ITEM_NODE_SUPPORTED_TGT_TYPES, 1, &devType);

  // Initialize node and RF4CE stack on the network processor
  printf("Calling RTI_InitReq\n");
  RTI_InitReq();

  // Use keypress to continue to pairing after RTI_InitCnf() is completed
  ch = _getch();

  // Request pairing
  printf("Calling RTI_PairReq\n");
  RTI_PairReq();

  // Use keypress to continue to data transfer after RTI_PairCnf() is completed
  while ((ch = _getch()) != 'q')
  {

    // Only send data if we have a valid pairing
    if (destIdx != RTI_INVALID_PAIRING_REF)
    {

      // Build CERC command message
      data[0] = RTI_CERC_USER_CONTROL_PRESSED;
      validKey = 0;


      // Map key input
      if (ch == '0')
      {
        data[1] = RTI_CERC_NUM_0;
        validKey = 1;
      }
      else if (ch == '1')
      {
        data[1] = RTI_CERC_NUM_1;
        validKey = 1;
      }
      else if (ch == '2')
      {
        data[1] = RTI_CERC_NUM_2;
        validKey = 1;
      }
      else if (ch == '3')
      {
        data[1] = RTI_CERC_NUM_3;
        validKey = 1;
      }
      else if (ch == '4')
      {
        data[1] = RTI_CERC_NUM_4;
        validKey = 1;
      }
      else if (ch == '5')
      {
        data[1] = RTI_CERC_NUM_5;
        validKey = 1;
      }
      else if (ch == '6')
      {
        data[1] = RTI_CERC_NUM_6;
        validKey = 1;
      }
      else if (ch == '7')
      {
        data[1] = RTI_CERC_NUM_7;
        validKey = 1;
      }
      else if (ch == '8')
      {
        data[1] = RTI_CERC_NUM_8;
        validKey = 1;
      }
      else if (ch == '9')
      {
        data[1] = RTI_CERC_NUM_9;
        validKey = 1;
      }

      // Only send data if '0'-'9' is pressed
      if (validKey)
      {
        // Wake up UART interface to send data.
        RTI_DisableSleepReq();
      }
    }


  }  // quit

  // Wake up RNP UART interface for next application run.
  RTI_DisableSleepReq();

  RTI_CloseWin32Module();
  return 0;
}


/**************************************************************************************************
 *
 * @fn      RTI_InitCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_InitReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_InitReq has returned.
 *
 * @param   status - Result of RTI_InitReq API call.
 *
 * @return  void
 */
void RTI_InitCnf(rStatus_t status)
{
  printf("RTI_InitCnf called with status %u\n", (unsigned) status);
  printf("Press any key to continue to pairing\n");

}

/**************************************************************************************************
 *
 * @fn      RTI_PairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_PairReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_PairReq has returned.
 *
 * @param   status - Result of RTI_PairReq API call.
 * @param   dstIndex - Pairing table index of paired device, or invalid.
 * @param   devType  - Pairing table index device type, or invalid.
 * @return  void
 */
void RTI_PairCnf( rStatus_t status, uint8 dstIndex, uint8 devType )
{
  // Set paring reference (destination index)
  destIdx = dstIndex;

  printf("RTI_PairCnf called with status %u\n", (unsigned) status);

  // Sleep the controller node UART interface.
  printf("RTI_EnableSleepReq....\n");
  RTI_EnableSleepReq();

}

/**************************************************************************************************
 *
 * @fn      RTI_AllowPairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_AllowPairReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_AllowPairReq has returned.
 * @param   status   - Result of RTI_PairReq API call.
 * @param   dstIndex - Pairing table entry of paired device, or invalid
 * @param   devType  - Pairing table index device type, or invalid
 *
 * @return  void
 */
void RTI_AllowPairCnf( rStatus_t status, uint8 dstIndex, uint8 devType )
{
}

// Tentative - this obsolete function won't be needed eventually.
void RTI_DiscoverCnf( rStatus_t status )
{
}

/**************************************************************************************************
 *
 * @fn      RTI_SendDataCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_SendDataReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_SendDataReq has returned.
 *
 * @param   status - Result of RTI_SendDataReq API call.
 *
 * @return  void
 */
void RTI_SendDataCnf( rStatus_t status )
{
  printf("RTI_SendDataCnf called with status %u\n", (unsigned) status);
  
  // Sleep RNP UART interface.
  printf("RTI_EnableSleepReq....\n");
  RTI_EnableSleepReq();

}

/**************************************************************************************************
 *
 * @fn      RTI_StandbyCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_StandbyReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_RxEnableReq has returned.
 *
 * input parameters
 *
 * @param   status - RTI_SUCCESS
 *                   RTI_ERROR_INVALID_PARAMETER
 *                   RTI_ERROR_UNSUPPORTED_ATTRIBUTE
 *                   RTI_ERROR_INVALID_INDEX
 *                   RTI_ERROR_UNKNOWN_STATUS_RETURNED
 *
 * output parameters
 *
 * None.
 *
 * @return  None.
 */
void RTI_StandbyCnf( rStatus_t status )
{
}

/**************************************************************************************************
 *
 * @fn      RTI_ReceiveDataInd
 *
 * @brief   RTI receive data indication callback asynchronously initiated by
 *          another node. The client is expected to complete this function.
 *
 * input parameters
 *
 * @param   srcIndex:  Pairing table index.
 * @param   profileId: Profile identifier.
 * @param   vendorId:  Vendor identifier.
 * @param   rxLQI:     Link Quality Indication.
 * @param   rxFlags:   Receive flags.
 * @param   len:       Number of bytes to send.
 * @param   *pData:    Pointer to data to be sent.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 */
void RTI_ReceiveDataInd( uint8 srcIndex, uint8 profileId, uint16 vendorId, uint8 rxLQI, uint8 rxFlags, uint8 len, uint8 *pData )
{
}

/**************************************************************************************************
 *
 * @fn      RTI_RxEnableCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_RxEnableReq API
 *          call. The client is expected to complete this function.
 *
 * @param   status - Result of RTI_EnableRxReqReq API call.
 *
 * @return  void
 */
void RTI_RxEnableCnf( rStatus_t status )
{
}

/**************************************************************************************************
 *
 * @fn      RTI_EnableSleepCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_EnableSleepReq API
 *          call. The client is expected to complete this function.
 *
 * @param   status - Result of RTI_EnableSleepReq API call.
 *
 * @return  void
 *
 */
void RTI_EnableSleepCnf( rStatus_t status )
{
 
  printf("RTI_EnableSleepCnf called with status %u\n", (unsigned) status);
  printf("Press '0'-'9' to send CERC command or 'q' to quit\n");
}


/**************************************************************************************************
 *
 * @fn      RTI_DisableSleepCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_DisableSleepReq API
 *          call. The client is expected to complete this function.
 *
 * @param   status - Result of RTI_EnableSleepReq API call.
 *
 * @return  void
 *
 */
void RTI_DisableSleepCnf( rStatus_t status )
{

  if ( ch != 'q')
  {
  // The RNP UART interface is awake, go ahead and send data
  printf("RTI_DisbleSleepCnf called with status %u\n", (unsigned) status);
  printf("Sending %c\n", ch);
  RTI_SendDataReq(destIdx,
    RTI_PROFILE_CERC,
    0, // vendor ID - not applicable in this case
    RTI_TX_OPTION_ACKNOWLEDGED,
    2,
    data);
  }

}

/**************************************************************************************************
 *
 * @fn      RTI_UnpairInd
 *
 * @brief   RTI indication callback initiated by receiving unpair request command.
 *
 * @param   dstIndex - Pairing table index of paired device.
 *
 * @return  void
 */
void RTI_UnpairInd( uint8 dstIndex )
{
	// This function is introduced in RemoTI 1.
}

/**************************************************************************************************
 *
 * @fn      RTI_PairAbortCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_PairAbortReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_PairAbortReq has returned.
 *
 * @param   status - Result of RTI_PairAbortReq API call.
 * @return  void
 */
void RTI_PairAbortCnf( rStatus_t status )
{
	// This function is introduced in RemoTI 1.
}

/**************************************************************************************************
 *
 * @fn      RTI_UnpairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_UnpairReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_UnpairReq has returned.
 *
 * @param   status   - Result of RTI_PairReq API call.
 * @param   dstIndex - Pairing table index of paired device, or invalid.
 *
 * @return  void
 */
void RTI_UnpairCnf( rStatus_t status, uint8 dstIndex )
{
  // This function is introduced in RemoTI 1.1
}

// Callback function for RCN applications
extern "C" void RCN_CbackEvent(void *pEvent)
{
}
