/**************************************************************************************************
    Filename:       npi_mspspi.c
    Revised:        $Date: 2009-02-25 06:05:23 -0800 (Wed, 25 Feb 2009) $
    Revision:       $Revision: 19258 $

    Description: This file contains the routines used to implement the Network
                 Processor Interface (NPI). It is an API for the inter-processor
                 communications between the Application Processor (master) and
                 the Network Processor (slave).

                 NOTE: The NPI supports a number of different physical layers,
                       such as SPI, UART, and I2C. Which NPI file compiled and
                       linked into the build will be determined by the build
                       process, thus avoiding the confusion of multiple ifdefs.

    Copyright 2008-2009 Texas Instruments Incorporated. All rights reserved.

    IMPORTANT: Your use of this Software is limited to those specific rights
    granted under the terms of a software license agreement between the user
    who downloaded the software, his/her employer (which must be your employer)
    and Texas Instruments Incorporated (the "License").  You may not use this
    Software unless you agree to abide by the terms of the License. The License
    limits your use, and you acknowledge, that the Software may not be modified,
    copied or distributed unless embedded on a Texas Instruments microcontroller
    or used solely and exclusively in conjunction with a Texas Instruments radio
    frequency transceiver, which is integrated into your product.  Other than for
    the foregoing purpose, you may not use, reproduce, copy, prepare derivative
    works of, modify, distribute, perform, display or sell this Software and/or
    its documentation for any purpose.

    YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
    PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
    INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
    NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
    TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
    NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
    LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
    INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
    OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
    OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
    (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

    Should you have any questions regarding your right to use this Software,
    contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

#include "hal_board.h"
#include "hal_rpc.h"
#include "npi.h"

/* ------------------------------------------------------------------------------------------------
 *                                           Externals
 * ------------------------------------------------------------------------------------------------
 */

extern void halResetSlave( void );

/* ------------------------------------------------------------------------------------------------
 *                                           Macros
 * ------------------------------------------------------------------------------------------------
 */

#define SRDY()                   ((SPI_SRDY_PORT & SPI_SRDY_BV) == 0)
#define _MRDY_Clr()              (SPI_MRDY_PORT |= SPI_MRDY_BV)
#define _MRDY_Set()              (SPI_MRDY_PORT &= ~SPI_MRDY_BV)
#define _SS_Clr()                (SPI_SS_PORT |= SPI_SS_BV)
#define _SS_Set()                (SPI_SS_PORT &= ~SPI_SS_BV)
#define MRDY_Clr()               st(_MRDY_Clr(); _SS_Clr();)
#define MRDY_Set()               st(_MRDY_Set(); _SS_Set();)

#define SS_Clr()
#define SS_Set()

// RPC protocol on SREQ is to wait for slave to signal ready by setting its SRDY line to opposite.
#define SRDY_RSP()  (!SRDY())

// Configure for ISR on falling edge: SRDY.
#define HAL_CFG_SRDY_ISR         (SPI_SRDY_IES |= SPI_SRDY_BV)

// Configure for ISR on rising edge: SRSP ready.
#define HAL_CFG_SRDY_RSP_ISR     (SPI_SRDY_IES &= ~SPI_SRDY_BV)

// POSSIBLY?
#define ASSERT_MRDY              st(MRDY_Set();)
#define DEASSERT_MRDY            st(MRDY_Clr();)
#define WHILE_SRDY_IS_ASSERTED   st(while(SRDY());)
#define WHILE_SRDY_IS_DEASSERTED st(while (!SRDY());)
#define IS_SRDY_DEASSERTED       st(!SRDY())
#define IS_SRDY_ASSERTED         st(SRDY())

/* ------------------------------------------------------------------------------------------------
 *                                           Constants
 * ------------------------------------------------------------------------------------------------
 */

#define SRDY_WAIT_MSECS  10

/* ------------------------------------------------------------------------------------------------
 *                                           Local Variables
 * ------------------------------------------------------------------------------------------------
 */
// NPI flags for processing events
volatile uint16 npiEventFlags;

// cache for saving SRSP buffer (UART only)
npiMsgData_t *pSynchReplyMsg;

// cache for saving AREQ buffer (UART only)
npiMsgData_t *pAsynchRequestMsg;

// synchronization flag for handling SREQ/SRSP processing (UART only)
volatile uint8 npiSynchReplyFlag;

// polling delay for waiting on SRSP (UART only)
uint32 npiSynchReplyDelay;

// data buffer
uint8 npi_spi_buf[AP_MAX_BUF_LEN];

/* ------------------------------------------------------------------------------------------------
 *                                           Local Functions
 * ------------------------------------------------------------------------------------------------
 */

// HAL SPI
void spiAREQ(uint8 *pBuf);
void spiSREQ(uint8 *pBuf);
void spiPOLL(uint8 *pBuf);

// System
void *msg_memcpy( void *dst, const void *src, uint16 len );

// NPI
static void  npiSpiInit( void );
static void  npiWrite( uint8 *pBuf, uint8 len );
static void  npiProcessMsg( uint8 *pMsg );
void  NPI_PollSlave( void );
void         npiSynchSlave( void );
static void  getSRDY1( void );
static uint8 getSRDY2( void );

/**************************************************************************************************
 * @fn          NPI_Init
 *
 * @brief       This function handles all NPI task initialization.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void NPI_Init( void )
{
  // setup USART1 as a 3-wire SPI master
  npiSpiInit();
}


/**************************************************************************************************
 * @fn          NPI_SendAsynchData API
 *
 * @brief       This function is called by the client when it has data ready to be sent
 *              asynchronously. This routine allocates an AREQ buffer, copies the client's
 *              payload, and sets up the send.
 *
 * input parameters
 *
 * @param *Msg  - Pointer to structure containing RPC message
 *
 * output parameters
 *
 * None.
 *
 * @return      None
 **************************************************************************************************
 */
void NPI_SendAsynchData( npiMsgData_t *pMsg )
{
  // add proper RPC type to header
  ((uint8 *)pMsg)[RPC_POS_CMD0] = (((uint8 *)pMsg)[RPC_POS_CMD0] & RPC_SUBSYSTEM_MASK) | RPC_CMD_AREQ;

  // send the message
  spiAREQ( (uint8 *)pMsg );
}


/**************************************************************************************************
 * @fn          NPI_SendSynchData
 *
 * @brief       This function is called by the client when it has data ready to 
 *              be sent synchronously. After the SREQ data is sent, this 
 *              routine waits for the SRSP reply, and puts the data in the same 
 *              buffer used for SREQ (and thus it must be large enough). When
 *              the interface link is SPI, this is done directly by polling
 *              SRDY. When the interface link is UART, the polling is done
 *              indirectly using a synchronization flag.
 *
 * input parameters
 *
 * @param       *pMsg - Pointer to the SREQ message data to be sent.
 *
 * output parameters
 *
 * @param       *pMsg - Pointer to SRSP message data received.
 *
 * @return      None.  
 **************************************************************************************************
 */
void NPI_SendSynchData( npiMsgData_t *pMsg )
{
  // add proper RPC type to header
  ((uint8 *)pMsg)[RPC_POS_CMD0] = (((uint8 *)pMsg)[RPC_POS_CMD0] & RPC_SUBSYSTEM_MASK) | RPC_CMD_SREQ;

  // send the SREQ message
  // Note that the SRSP is placed in this same buffer when this call returns
  spiSREQ( (uint8 *)pMsg );

  // this is the SRSP; clear out the RPC type in header
  ((uint8 *)pMsg)[RPC_POS_CMD0] &= RPC_SUBSYSTEM_MASK;
}

/**************************************************************************************************
 * @fn          NPI_PollSlave
 *
 * @brief       Poll the NPI-only slave for asynchronous command and drive the RPC transaction.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void NPI_PollSlave(void)
{
  // If the NP slave indicates that an AREQ is ready, then poll for it.
  // When SRDY is asserted, the master sends a POLL command, then waits for SRDY to deassert. Then
  // the master reads the slave's data.
  if (SRDY()) // SRDY == 0
  {
    uint8 *pBuf = npi_spi_buf;

    // send POLL command and return slave's AREQ message in pBuf
    spiPOLL(pBuf);

    // setup callback to client for message processing
    // Note: note that the slave's AREQ is basically a SRSP from the POLL (which is like a SREQ)
    npiProcessMsg( pBuf );
  }
}


/**************************************************************************************************
 * @fn          npiProcessMsg
 *
 * @brief       This function preps a valid RPC message for a NPI client
 *              callback for payload processing.
 *
 * input parameters
 *
 * @param       pMsg - Pointer to the start of a valid RPC message.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void npiProcessMsg( uint8 *pMsg )
{
  // only AREQ is supported from slave in SPI
  if ( (pMsg[RPC_POS_CMD0] & RPC_CMD_TYPE_MASK) == RPC_CMD_AREQ )
  {
    // remove type from header
    pMsg[RPC_POS_CMD0] &= RPC_SUBSYSTEM_MASK;

    // callback to client
    NPI_AsynchMsgCback( (npiMsgData_t *)pMsg );
  }
}


/**************************************************************************************************
 * @fn          spiAREQ
 *
 * @brief       This function effects an asynchronous transaction with the NP 
 *              slave according to the RPC protocol for the AREQ.
 *
 * input parameters
 *
 * @param       pBuf - Pointer to the buffer to Tx across the SPI.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void spiAREQ(uint8 *pBuf)
{
  do {
    getSRDY1(); // doesn't return until SRDY has been asserted, even if this required a CC2430EM reset!
    // Send the host AREQ-message across SPI.
    npiWrite(pBuf, (*pBuf+RPC_FRAME_HDR_SZ));
  } while (!getSRDY2());

  MRDY_Clr();                     // MRDY must be cleared before setting it again to talk again.
}

/**************************************************************************************************
 * @fn          spiSREQ
 *
 * @brief       This function effects a synchronous transaction with the NP slave
 *              according to the RPC protocol for the SREQ.
 *
 * input parameters
 *
 * @param       pBuf - Pointer to the buffer to Tx across the SPI.
 *
 * output parameters
 *
 * @param       pBuf - Pointer to the data received across the SPI.
 *
 * @return      None.
 **************************************************************************************************
 */
void spiSREQ(uint8 *pBuf)
{
  do {
    // assert MRDY then wait for SRDY to be asserted (power save while waiting, or timeout and reset slave)
    getSRDY1();

    // Send the host SREQ-message across SPI.
    // Note: This could be a POLL command with no data, or a SREQ with a possible payload
    // Note: Frame header (length and command) size is 3, so length is payload length plus 3
    npiWrite(pBuf, (*pBuf+RPC_FRAME_HDR_SZ));

    // now wait until SRDY is deasserted indicating it is ready to send the SRSP
  } while (!getSRDY2()); // wait for SRDY==1

  // moved this code
  /* Now setup the POLL command in the buffer which has just been transmitted and which will now
   * be used to receive the SREQ in response.
   */

  // Arbitrary data is built to be transmitted just so that the SPI clock is fed to
  // the slave. Data is cleared just so that slave does not receive and handle the data
  // as recognized as real command.
  pBuf[0] = 0;
  pBuf[1] = 0;
  pBuf[2] = 0;

  // Arbitrary data writing triggers clock feed 
  // in order to receive from the slave (since the master controls the clock)
  npiWrite(pBuf, RPC_FRAME_HDR_SZ);  // Send the POLL across the SPI.

  // Receive the rest of the slave's message if it is longer than the POLL.
  // Note: The first byte at *pBuf contains the length of the payload
  if (*pBuf != 0)
  {
    // write the number of bytes of payload to get that number of bytes back from the slave
    // Note: we are starting at offset 3 since the header was already received
    // NOTE: FOR SOME REASON THE DMA SENDS N-1 BYTES TO THE MASTER (THE MASTER
    //       ALWAYS GETS ZERO ON THE LAST PAYLOAD BYTE). NOT SURE WHY, BUT FOR
    //       NOW, THE SLAVE WILL ADD ONE EXTRA BYTE TO PAYLOAD AND PAYLOAD LENGTH

    // The following zero initialization seems to help slave to complete DMA transaction,
    // Without zeroing out, slave occasionally gets stuck without completing DMA
    // transfer.
    {
      uint8 i;
      for (i = 0; i < *pBuf; i++)
      {
        pBuf[RPC_FRAME_HDR_SZ + i] = 0;
      }
    }
    npiWrite(pBuf+RPC_FRAME_HDR_SZ, *pBuf);
  }

  MRDY_Clr();  // MRDY must be cleared before setting it again to talk again.
}

/**************************************************************************************************
 * @fn          spiPOLL
 *
 * @brief       This function polls the NP slave according to the RPC protocol for POLL.
 *
 * input parameters
 *
 * @param       pBuf - Pointer to the buffer to Tx a POLL across the SPI and then receive the AREQ.
 *
 * output parameters
 *
 * @param       pBuf - Pointer to the buffer to Tx a POLL across the SPI and then receive the AREQ.
 *
 * @return      None.
 **************************************************************************************************
 */
void spiPOLL(uint8 *pBuf)
{
  // Setup the POLL command in the buffer.
  *pBuf = 0;                      // POLL command has zero data bytes.
  *(pBuf+1) = RPC_CMD_POLL;       // POLL command MSB.
  *(pBuf+2) = 0;                  // POLL command LSB.

  // send POLL command, wait for SRDY to go high, then read the slave's AREQ message
  spiSREQ(pBuf);
}

/**************************************************************************************************
 * @fn          getSRDY1
 *
 * @brief       This function asserts MRDY and waits for SRDY asertion.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void getSRDY1(void)
{
  MRDY_Set();  // MRDY must be set before talking to the slave.

  // A RACE CONDITION EXISTS HERE WHEN SLAVE REQUESTED POLL AS SRDY IS ALREADY
  // ASSERTED. SO THIS DELAY HAD BETTER BE LONG ENOUGH FOR THE SLAVE TO SERVICE
  // THE MRDY ISR, AND SETUP THE RX DMA!!
  halDelay(SRDY_WAIT_MSECS, FALSE);

  while (!SRDY() && !halDelayDone()) // wait while SRDY==1 and delay not done
  {
    HAL_LOW_POWER_MODE(); // SRDY high-to-low transition causes interrupt that takes you out of low power mode
  }

  while(!SRDY());

  if (!SRDY()) // SRDY is still 1 so reset the slave!
  {
    // we are here because we timed out
    MRDY_Clr();  // MRDY must be cleared before setting it again to talk again.
    halResetSlave(); // reset the slave
    npiSynchSlave(); // WAIT UNTIL SRDY IS ASSERTED (FOREVER)!!
    MRDY_Set();  // MRDY must be set before talking to the slave.
  }

  // ELSE SRDY HAS BEEN ASSERTED (i.e. SRDY==0)
}

/**************************************************************************************************
 * @fn          getSRDY2
 *
 * @brief       This function times out waiting for the SRDY_RSP.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      TRUE if the slave signals SRDY_RSP.
 **************************************************************************************************
 */
static uint8 getSRDY2(void)
{
  // Slave may set SRDY high to acknowledge reception before MSP rising edge interrupt logic
  // is ready to be fired and then before clock tick time (10ms) client may attempt to send
  // AREQ data which will cause SRDY to be asserted (0) again.
  // In this case 10ms sampling will not catch such transition.
  // Hence MSP should not enter power saving mode and that means no need to set up interrupt
  // for rising edge of SRDY.
  
  // generate interrupt on rising edge of SRDY
  // the interrupt based on the SRDY transition is defined here so that we can emerge from sleep
  //HAL_CFG_SRDY_RSP_ISR;

  // wait 10ms before timeout
  halDelay(SRDY_WAIT_MSECS, FALSE);

  // while SRDY==0 and not timed out yet, sleep (SRDY or timeout interrupt will awaken us)
  while (!SRDY_RSP() && !halDelayDone())
  {
    //HAL_LOW_POWER_MODE();
  }

  // generate interrupt on falling edge of SRDY
  // the interrupt based on the SRDY transition is defined here so that we can emerge from sleep
  //   and the next check of getSRDY1()
  //HAL_CFG_SRDY_ISR;

  return SRDY_RSP(); // SRDY==0 if timed out (return FALSE), SRDY==1 otherwise (return TRUE)
}


/**************************************************************************************************
 * @fn          npiSpiInit
 *
 * @brief       This function initializes the SPI.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void npiSpiInit(void)
{
  // Reset and configure mode register for SPI
  SPI_RESET_MODE();
  
  // Configure IO pins
  
  /* MOSI - Peripheral/Output */
  SPI_MOSI_SEL |= SPI_MOSI_BV;
  SPI_MOSI_DDR |= SPI_MOSI_BV;

  /* MISO - Peripheral/Input */
  SPI_MISO_SEL |= SPI_MISO_BV;
  SPI_MISO_DDR &= ~SPI_MOSI_BV;

  /* CLK - Peripheral/Output */
  SPI_CLK_SEL |= SPI_CLK_BV;
  SPI_CLK_DDR |= SPI_CLK_BV;

  /* SS - I/O; Output */
  SPI_SS_SEL &= ~SPI_SS_BV;
  SPI_SS_DDR |= SPI_SS_BV;

  // Note that interrupt is not enabled
  
  // Pull out of reset
  SPI_CLEAR_RESET();
}

/**************************************************************************************************
 * @fn          npiWrite
 *
 * @brief       This function blocks (polling) on the SPI port; USART by ISR.
 *
 * input parameters
 *
 * @param       pBuf - Pointer to the buffer that contains the data to transmit.
 * @param       len - Length of the data to transmit.
 *
 * output parameters
 *
 * @param       pBuf - SPI only: Pointer to the buffer that is filled with the Rx bytes.
 *
 * @return      None.
 **************************************************************************************************
 */
void npiWrite(uint8 *pBuf, uint8 len)
{
    SS_Set();

    while (len--)
    {
      // write the first byte the USART1 output port as a SPI byte
      SPI_TXBUF = *pBuf;

      // wait until tx byte is scanned to slave, which clocks slave data out back to the master
      SPI_WAIT_RX();

      // got the received byte, so save it and advanced the pointer to send the next byte
      *pBuf++ = SPI_RXBUF;
    }

    SS_Clr();
}

/**************************************************************************************************
 * @fn          npiSynchSlave
 *
 * @brief       This function performs handshake with slave to synchronize with slave after
 *              reset.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void npiSynchSlave( void )
{
  // wait until SRDY is asserted (i.e. SRDY==0)
  while (!SRDY());  // Wait for the Z-Accel slave to signal that it is out of reset.
  //WHILE_SRDY_IS_DEASSERTED;

  // master asserts MRDY to begin syncrhonization
  // note that after init, the slave will treat the first assertion of MRDY as a synchronization signal
  MRDY_Set();
  //ASSERT_MRDY;

  // master waits until slave detects MRDY assertion, and deasserts SRDY in response
  while(SRDY()); // wait while SRDY==0
  //WHILE_SRDY_IS_ASSERTED;

  // slave has deasserted SRDY so master deasserts MRDY
  MRDY_Clr();
  //DEASSERT_MRDY;

  // synchronization is complete!
}

/*********************************************************************
 * @fn      msg_memcpy
 *
 * @brief
 *
 *   Generic memory copy.
 *
 *   Note: This function differs from the standard memcpy(), since
 *         it returns the pointer to the next destination byte. The
 *         standard memcpy() returns the original destination address.
 *
 * @param   dst - destination address
 * @param   src - source address
 * @param   len - number of bytes to copy
 *
 * @return  pointer to end of destination buffer
 */
void *msg_memcpy( void *dst, const void *src, uint16 len )
{
  uint8 *pDst;
  const uint8 *pSrc;

  pSrc = src;
  pDst = dst;

  while ( len-- )
    *pDst++ = *pSrc++;

  return ( pDst );
}


/**************************************************************************************************
 **************************************************************************************************/
