/**************************************************************************************************
    Filename:       ap_main.c
    Revised:        $Date: 2009-03-03 09:02:53 -0800 (Tue, 03 Mar 2009) $
    Revision:       $Revision: 19294 $

    Description:

    This file contains Host Processor Sample Application (SA) to exercise the
    RemoTI (RTI) Surrogate (RTIS) API interface to the Network Processor (NP)
    using the Network Processor Interface (NPI).

    Copyright 2008-2009 Texas Instruments Incorporated. All rights reserved.

    IMPORTANT: Your use of this Software is limited to those specific rights
    granted under the terms of a software license agreement between the user
    who downloaded the software, his/her employer (which must be your employer)
    and Texas Instruments Incorporated (the "License").  You may not use this
    Software unless you agree to abide by the terms of the License. The License
    limits your use, and you acknowledge, that the Software may not be modified,
    copied or distributed unless embedded on a Texas Instruments microcontroller
    or used solely and exclusively in conjunction with a Texas Instruments radio
    frequency transceiver, which is integrated into your product.  Other than for
    the foregoing purpose, you may not use, reproduce, copy, prepare derivative
    works of, modify, distribute, perform, display or sell this Software and/or
    its documentation for any purpose.

    YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
    PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
    INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
    NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
    TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
    NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
    LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
    INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
    OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
    OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
    (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

    Should you have any questions regarding your right to use this Software,
    contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

// C Std.
#include "string.h"

// HAL includes
#include "hal_board.h"
#include "hal_lcd.h"

// RTI includes
#include "rti.h"

// SA includes
#include "ap_main.h"

/* ------------------------------------------------------------------------------------------------
 *                                           Externals
 * ------------------------------------------------------------------------------------------------
 */
// NPI MSP implementation specific call
extern void NPI_PollSlave( void );

#if ((defined HAL_UART) && (HAL_UART == TRUE))

  // Client Callback
  extern void halUartReqCback( uint16 event );

#endif

/* ------------------------------------------------------------------------------------------------
 *                                           Constants
 * ------------------------------------------------------------------------------------------------
 */

// Application States
enum
{
  AP_STATE_INIT,  // Initial state - prior to initialization completion
  AP_STATE_READY, // Ready state - ready to pair, ready to send data
  AP_STATE_PAIR,  // Pairing in progress state
  AP_STATE_NDATA, // Send-data in progress state
  AP_STATE_WAIT_WAKEUP_FOR_PAIR, // Waiting for RTI wakeup from sleep to do pairing
  AP_STATE_WAIT_WAKEUP_FOR_NDATA // Waiting for RTI wakeup from sleep to send data
};

// LED Blinking states
enum
{
  AP_NO_BLINK, // blinking is turned off
  AP_BLINK_LED_ON, // blinking is on and LED is turned on at the moment
  AP_BLINK_LED_OFF // blinking is on and LED is turned off at the moment
};

/* ------------------------------------------------------------------------------------------------
 *                                           Typedefs
 * ------------------------------------------------------------------------------------------------
 */

/* ------------------------------------------------------------------------------------------------
 *                                           Macros
 * ------------------------------------------------------------------------------------------------
 */
// Fatal error handling macro : reusing RTIS fatal error callback function
#define AP_FATAL_ERROR() rtisFatalError(0xff)

/* ------------------------------------------------------------------------------------------------
 *                                           Global Variables
 * ------------------------------------------------------------------------------------------------
 */

/* ------------------------------------------------------------------------------------------------
 *                                           Local Variables
 * ------------------------------------------------------------------------------------------------
 */

// Task Related
volatile uint16 apEvtFlags;

// Application state variable
static uint8 apState;

// current target device destination index
uint8 apDestIdx;

// LED blinking status
static uint8 apBlinkRed = AP_NO_BLINK;

// pairing reference entry buffer
static rcnNwkPairingEntry_t apPairingEntryBuf;

/* ------------------------------------------------------------------------------------------------
 *                                           Local Functions
 * ------------------------------------------------------------------------------------------------
 */

// HAL task event handler
static uint16 HAL_ProcessEvent( void );

// sample application tasks Related
static void   apInit( void );
static uint16 AP_ProcessEvent( void );

// Utility function to set event flag
void setEventFlag( volatile uint16 *event_flag, uint16 event );
void clearEventFlag( volatile uint16 *event_flag, uint16 event );

void apConfigParam( void );
void apGenerateNextKeyPress( void );
static void apDispHoldMsg( void );
static void apDispInitMsg( void );
static void apDispPairMsg( void );
static void apDispPairedMsg( void );
static void apDispFailedMsg( void );

// Implementation of RTIS fatal error handling
void rtisFatalError( uint8 status );

/**************************************************************************************************
 * @fn          main
 *
 * @brief       This is the C-code entry function after powerup and compiler generated inits.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void main(void)
{
  // perform DIO init, MCU init, Timer init, and SPI init
  halBoardInit();

  HAL_ENABLE_INTERRUPTS();

  HAL_TURN_OFF_RED();

  // initialize RemoTI (note, taskId not used on AP)
  RTI_Init( 0 );
  
  // display hold message
  apDispHoldMsg();

  // start periodic timer
  halTimerSet( HAL_IDX_TIMER_LED, SA_BLINK_INTERVAL, HAL_TIMER_AUTO );

  // sample app init
  apInit();

  // Executive Loop: Order of process event calls dictates task priority
  while(TRUE)
  {
    // Process all HAL related events
    while( HAL_ProcessEvent() );

    // Poll SRDY line and take actions accordingly
    NPI_PollSlave();

    // Process all Sample Application related events
    while( AP_ProcessEvent() );

#ifdef POWER_SAVING
    // if all task events have been processed, enter low power mode
    HAL_LOW_POWER_MODE();
#endif
  }
}

/**************************************************************************************************
 * @fn          apInit
 *
 * @brief       This function is the sample application initialization routine.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void apInit( void )
{
  // set sample application state to not ready until RTI Init is successfully confirmed
  apState = AP_STATE_INIT;

  // schedule the SA to send the first message (no critical section required)
  apEvtFlags |= SA_EVT_INIT;
}


/**************************************************************************************************
 * @fn          HAL_ProcessEvent
 *
 * @brief       This function is used to process HAL events.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      TRUE if a HAL event was processed; FALSE otherwise.
 **************************************************************************************************
 */
uint16 HAL_ProcessEvent( void )
{
  uint16 event = HAL_EVT_NONE;

  if (halEventFlags & HAL_EVT_BTN1_PRESS)
  {
    // Switch 1 is pressed
    
    event = HAL_EVT_BTN1_PRESS;
    
    // use switch 1 keypress for pairing
    if (apState == AP_STATE_READY)
    {
      // Change state    
      apState = AP_STATE_WAIT_WAKEUP_FOR_PAIR;
      
      // Start blinking Red LED
      HAL_TURN_ON_RED();
      apBlinkRed = AP_BLINK_LED_ON;
      
      // wakeup the NP from sleep so we can get to work
      RTI_DisableSleepReq();
    }
  }
  else if (halEventFlags & HAL_EVT_BTN2_PRESS)
  {
    // Switch 2 is pressed
    
    event = HAL_EVT_BTN2_PRESS;
    
    // use switch 2 keypress for sending data
    // sequence through every numerical key on the MS Media Center RC
    apGenerateNextKeyPress();
  }
  else if ( halEventFlags & HAL_EVT_TIMER_LED )
  {
    // Periodic timer event for LED blinking and LCD updates
    
    event = HAL_EVT_TIMER_LED;

    // blink LEDs if necessary
    if (apBlinkRed == AP_BLINK_LED_ON)
    {
      HAL_TURN_OFF_RED();
      apBlinkRed = AP_BLINK_LED_OFF;
    }
    else if (apBlinkRed == AP_BLINK_LED_OFF)
    {
      HAL_TURN_ON_RED();
      apBlinkRed = AP_BLINK_LED_ON;
    }
  }
  else if ( halEventFlags & HAL_EVT_JOY_LEFT )
  {
    event = HAL_EVT_JOY_LEFT;
    HalLcdWriteString("Left", HAL_LCD_LINE_3);
  }
  else if ( halEventFlags & HAL_EVT_JOY_RIGHT )
  {
    event = HAL_EVT_JOY_RIGHT;
    HalLcdWriteString("Right", HAL_LCD_LINE_3);
  }
  else if ( halEventFlags & HAL_EVT_JOY_SELECT )
  {
    event = HAL_EVT_JOY_SELECT;
    HalLcdWriteString("Select", HAL_LCD_LINE_3);
  }
  else if ( halEventFlags & HAL_EVT_JOY_UP )
  {
    event = HAL_EVT_JOY_UP;
    HalLcdWriteString("Up", HAL_LCD_LINE_3);
  }
  else if ( halEventFlags & HAL_EVT_JOY_DOWN )
  {
    event = HAL_EVT_JOY_DOWN;
    HalLcdWriteString("Down", HAL_LCD_LINE_3);
  }

  // clear HAL event flags in a critical section as they are set by interrupt handlers
  if (event != HAL_EVT_NONE)
  {
    clearEventFlag( &halEventFlags, event );
    return TRUE;
  }

  return FALSE;
}


/**************************************************************************************************
 * @fn          AP_ProcessEvent
 *
 * @brief       This function is used to process Sample Application events.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      TRUE if an event was processed; FALSE otherwise.
 **************************************************************************************************
 */
uint16 AP_ProcessEvent( void )
{
  uint16 event = SA_EVT_NONE;

  // scheduled by receipt of slave AREQ
  if (apEvtFlags & SA_EVT_INIT)
  {
    event = SA_EVT_INIT;
    
    // initialize parameters and start the RTI network
    apConfigParam();
    
    // indicate we are about to init the RTI
    apDispInitMsg();
   
    // initialize the RF4CE software and begin network operation
    // NOTE: Configuration Parameters need to be set prior to this call!
    RTI_InitReq();
  }

  // tidy up
  if (event != SA_EVT_NONE)
  {
    // elimiante processed events; no critical section required
    apEvtFlags ^= event;
    return TRUE;
  }

  return FALSE;
}


/**************************************************************************************************
 *
 * @fn      RTI_InitCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_InitReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_InitReq has returned.
 *
 * @param   status - Result of RTI_InitReq API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_InitCnf( rStatus_t status )
{
  {
    uint8 startupFlg;
    
    // In case startup flag was set to clear_state, change the startup flag to restore state.
    startupFlg = RESTORE_STATE;
    RTI_WriteItem( RTI_CP_ITEM_STARTUP_CTRL, 1, &startupFlg );
  }
  
  {
    uint8 i, count;
    
    // get the current pairing table index, if there is one
    apDestIdx = RTI_INVALID_PAIRING_REF;
    RTI_ReadItem( RTI_CONST_ITEM_MAX_PAIRING_TABLE_ENTRIES, 1, &count);
    for (i = 0; i < count; i++)
    {
      if (RTI_WriteItem( RTI_SA_ITEM_PT_CURRENT_ENTRY_INDEX, 1, &i) != RTI_SUCCESS)
      {
        // Problem in pairing table reading
        AP_FATAL_ERROR();
      }
      if (RTI_ReadItem( RTI_SA_ITEM_PT_CURRENT_ENTRY, sizeof(apPairingEntryBuf),
                       (uint8 *) &apPairingEntryBuf ) == RTI_SUCCESS)
      {
        // Reading current pairing table entry will succeed only when the entry corresponding
        // to the current index is valid.
        
        apDestIdx = i;
        // Continue till finding the last one, making the logic to select the last paired
        // entry (as far as none of the pairing entry is removed).
        // The actual application logic to selecting default destination will be more
        // complicated than this.
      }
    }
  }
  
  if ( status == RTI_SUCCESS )
  {
    // RTI has been successfully started, so application is now ready to go
    apState = AP_STATE_READY;
        
    if (apDestIdx == RTI_INVALID_PAIRING_REF)
    {
      // indicate we are ready for pairing
      apDispPairMsg();
    }
    else
    {
      // indicate we already have a pairing
      HAL_TURN_ON_RED();
      apDispPairedMsg();
    }
    
    // allow NP to sleep again
    RTI_EnableSleepReq();

  }
  else
  {
    apDispFailedMsg();
  }
}


/**************************************************************************************************
 *
 * @fn      RTI_PairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_PairReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_PairReq has returned.
 *
 * @param   status   - Result of RTI_PairReq API call.
 * @param   dstIndex - Pairing table index of paired device, or invalid.
 * @param   devType  - Pairing table index device type, or invalid.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_PairCnf( rStatus_t status, uint8 dstIndex, uint8 devType )
{
  uint8 numActiveEntries;
  
  // Unused argument reference to remove compiler warning
  (void) devType;
  
  if ( status == RTI_SUCCESS )
  {
    // Set destination index to the newly paired entry
    apDestIdx = dstIndex;
  }
  else
  {
    // Display pairing failure for a while
    apDispFailedMsg();
    halDelay(250, TRUE);
    halDelay(250, TRUE);
    halDelay(250, TRUE);
    halDelay(250, TRUE);
  }
  
  // Stop blinking RED LED
  apBlinkRed = AP_NO_BLINK;

  // Update LCD and LEDs for pairing status
  if (RTI_ReadItem(RTI_SA_ITEM_PT_NUMBER_OF_ACTIVE_ENTRIES, 1, &numActiveEntries) != RTI_SUCCESS)
  {
    AP_FATAL_ERROR();
  }
  else if (numActiveEntries > 0)
  {
    // indicate at least one valid pairing entry exists
    apDispPairedMsg();
      
    HAL_TURN_ON_RED();
  }
  else
  {
    // No pairing entry exists
    apDispPairMsg();
    HAL_TURN_OFF_RED();
  }

  // allow NP to sleep again
  RTI_EnableSleepReq();
  
  // Update application state
  apState = AP_STATE_READY;
}


/**************************************************************************************************
 *
 * @fn      RTI_SendDataCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_SendDataReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_SendDataReq has returned.
 *
 * @param   status - Result of RTI_SendDataReq API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_SendDataCnf( rStatus_t status )
{
  HAL_TURN_OFF_GRN();

  if ( status == RTI_SUCCESS )
  {
    if (apState == AP_STATE_NDATA)
    {
      apState = AP_STATE_READY;
    }
  }
  else
  {
      // let us keep going
      apState = AP_STATE_READY;    
  }
  
  // allow NP to sleep again
  RTI_EnableSleepReq();
}


/**************************************************************************************************
 *
 * @fn      RTI_EnableSleepCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_EnableSleepReq
 *          API call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_SendDataReq has returned.
 *
 * @param   status - Result of RTI_EnableSleepReq API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_EnableSleepCnf( rStatus_t status )
{
  if ( status != RTI_SUCCESS )
  {
    AP_FATAL_ERROR();    
  }
  apState = AP_STATE_READY;
}


/**************************************************************************************************
 *
 * @fn      RTI_DisableSleepCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_DisableSleepReq
 *          API call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_DisableSleepCnf has returned.
 *
 * @param   status - Result of RTI_DisableSleepCnf API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_DisableSleepCnf( rStatus_t status )
{
  static uint8 cmd = RTI_CERC_NUM_0;
  
  if ( status == RTI_SUCCESS )
  {
    if (apState == AP_STATE_WAIT_WAKEUP_FOR_PAIR)
    {
      // Start pairing
      apState = AP_STATE_PAIR;
      RTI_PairReq();
      return;
    }
    else if (apState == AP_STATE_WAIT_WAKEUP_FOR_NDATA)
    {
      // Send data
      if (apDestIdx != RTI_INVALID_PAIRING_REF)
      {
        uint8  profileId;
        uint16 vendorId;
        uint8  txOptions;
        uint8  len = 2;
        uint8  pData[5];
        
        // ready to send a CERC command
        if ( ++cmd > RTI_CERC_NUM_9 )
        {
          cmd = RTI_CERC_NUM_0;
        }
        
        // fist payload item is the key command
        pData[0] = RTI_CERC_USER_CONTROL_PRESSED;
        pData[1] = cmd;
        
        // sending data state
        apState = AP_STATE_NDATA;
        
        // send the data over the air
        txOptions = RTI_TX_OPTION_ACKNOWLEDGED;
        profileId = RTI_PROFILE_CERC;
        vendorId  = RTI_VENDOR_TEXAS_INSTRUMENTS;
        
        // send the keypress
        RTI_SendDataReq( apDestIdx, profileId, vendorId, txOptions, len, pData );
    
        // TURN ON LED 
        //HalLedSet(HAL_LED_1, HAL_LED_MODE_ON);
        HAL_TURN_ON_GRN();    
        return;
      }
    }
  }
  // No action taken. Go back to sleep
  RTI_EnableSleepReq();
  apState = AP_STATE_READY;    
}


// -- other unused callback functions --

void RTI_AllowPairCnf( rStatus_t status, uint8 dstIndex, uint8 devType )
{
  // Unused argument reference to remove compiler warning
  (void) status;
  (void) dstIndex;
  (void) devType;
}

void RTI_StandbyCnf( rStatus_t status )
{
  // Unused argument reference to remove compiler warning
  (void) status;
}

void RTI_ReceiveDataInd( uint8 srcIndex, uint8 profileId, uint16 vendorId, uint8 rxLQI, uint8 rxFlags, uint8 len, uint8 *pData )
{
  // Unused argument reference to remove compiler warning
  (void) srcIndex;
  (void) profileId;
  (void) vendorId;
  (void) rxLQI;
  (void) rxFlags;
  (void) len;
  (void) pData;
}

void RTI_RxEnableCnf( uint8 status )
{
  // Unused argument reference to remove compiler warning
  (void) status;
}

/*********************************************************************
 * @fn      setEventFlag
 *
 * @brief
 *
 *    This function is called to set the event flag for a task.  The
 *    event passed in is OR'd into the task's event variable.
 *
 * @param   byte event_flag - task's event flag
 * @param   byte event      - event to set
 *
 * @return
 *
 * None
 */
void setEventFlag( volatile uint16 *event_flag, uint16 event )
{
  halIntState_t intState;

  HAL_ENTER_CRITICAL_SECTION( intState );    // Hold off interrupts
  *event_flag |= event;                      // Stuff the event bit(s)
  HAL_EXIT_CRITICAL_SECTION( intState );     // Release interrupts
}

/*********************************************************************
 * @fn      clearEventFlag
 *
 * @brief
 *
 *    This function is called to clear the event flag for a task.  The
 *    event passed in is XOR'd into the task's event variable.
 *
 * @param   byte event_flag - task's event flag
 * @param   byte event      - event to clear
 *
 * @return
 *
 * None
 */
void clearEventFlag( volatile uint16 *event_flag, uint16 event )
{
  halIntState_t intState;

  HAL_ENTER_CRITICAL_SECTION( intState );    // Hold off interrupts
  *event_flag ^= event;                      // clear the event bit(s)
  HAL_EXIT_CRITICAL_SECTION( intState );     // Release interrupts
}


/**************************************************************************************************
 *
 * @fn      apConfigParam
 *
 * @brief   This function sets up the RTI configuration parameters.
 *
 * @param   
 *
 * None.
 *
 * @return  void
 *
 **************************************************************************************************/
void apConfigParam( void )
{
  uint8 pValue[MAX_AVAIL_DEVICE_TYPES]; // space for largest number of bytes, not counting strings
  
  //////////////////////////////
  // Configuration Parameters //
  //////////////////////////////
  
  // Set up configuration parameters that are different from default values

  // -- device type et al. settings. --
  
  // Set node capability settings value: Controller node supporting security
  pValue[0] = RTI_BUILD_NODE_CAPABILITIES(0, 0, 1, 0);
  
  // Write capabilities setting to RTI
  if (RTI_WriteItem( RTI_CP_ITEM_NODE_CAPABILITIES, 1, pValue) != RTI_SUCCESS)
  {
    AP_FATAL_ERROR();
  }
  
  // supported target type - This is just an example list.
  pValue[0] = RTI_DEVICE_TELEVISION;
  pValue[1] = RTI_DEVICE_VIDEO_PLAYER_RECORDER;
  pValue[2] = RTI_DEVICE_SET_TOP_BOX;
  if (RTI_WriteItem( RTI_CP_ITEM_NODE_SUPPORTED_TGT_TYPES, 3, pValue ) != RTI_SUCCESS)
  {
    AP_FATAL_ERROR();
  }
  
  if (halResetKeyState)
  {
    // User requested reset (state values reset)
    pValue[0] = CLEAR_STATE;
    (void) RTI_WriteItem( RTI_CP_ITEM_STARTUP_CTRL, 1, pValue );
  }
}


/**************************************************************************************************
 * @fn          apGenerateNextKeyPress
 *
 * @brief       This function generates virtual key presses.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void apGenerateNextKeyPress( void )
{
  if ( (apDestIdx != RTI_INVALID_PAIRING_REF) && (apState == AP_STATE_READY) )
  {
    // wakeup the NP from sleep so we can get to work
    apState = AP_STATE_WAIT_WAKEUP_FOR_NDATA;
    
    RTI_DisableSleepReq();
  }
}

/**************************************************************************************************
 * @fn          apDispHoldMsg
 *
 * @brief       This function displays HOLD on LCD.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void apDispHoldMsg( void )
{
  HalLcdWriteString("Hold", HAL_LCD_LINE_2);
}


/**************************************************************************************************
 * @fn          apDispHoldMsg
 *
 * @brief       This function displays PAIR on LCD.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void apDispPairMsg( void )
{
  HalLcdWriteString("Pair", HAL_LCD_LINE_2);
}


/**************************************************************************************************
 * @fn          apDispHoldMsg
 *
 * @brief       This function displays PAIRED on LCD.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void apDispPairedMsg( void )
{
  HalLcdWriteString("Paired", HAL_LCD_LINE_2);
}

/**************************************************************************************************
 * @fn          apDispHoldMsg
 *
 * @brief       This function displays FAILED on LCD.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void apDispFailedMsg( void )
{
  HalLcdWriteString("Failed", HAL_LCD_LINE_2);
}


/**************************************************************************************************
 * @fn          apDispHoldMsg
 *
 * @brief       This function displays INIT on LCD.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void apDispInitMsg( void )
{
  HalLcdWriteString("Init", HAL_LCD_LINE_2);
}


/**************************************************************************************************
 * @fn          rtisFatalError
 *
 * @brief       This function handles fatal error of RTI surrogate. The function implementation is
 *              platform and application specific.
 *
 * input parameters
 *
 * @param       status - error status
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void rtisFatalError( uint8 status )
{
  volatile uint16 i;

  // Unused argument reference to remove compiler warning
  (void)status;
  
  apDispFailedMsg();
  
  while(TRUE)
  {
    HAL_TURN_ON_RED();
    HAL_TURN_ON_GRN();
    i = 1000;
    while (--i > 0);
    HAL_TURN_OFF_RED();
    HAL_TURN_OFF_GRN();
    i = 1000;
    while (--i > 0);
  }
}


/**************************************************************************************************
 **************************************************************************************************/
