/*
 *  ======== rotate.h ========
 *  The rotate API.
 */

#ifndef extensions_rotate_ROTATE_H_
#define extensions_rotate_ROTATE_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <ti/sdo/ce/Engine.h>
#include <ti/sdo/ce/visa.h>
#include <ti/sdo/ce/skel.h>
#include <ti/xdais/xdas.h>
#include <ti/sdo/apps/codecs/rotate/irotate/adapter/irotate_adapt.h>

#define ROTATE_EOK      0
#define ROTATE_EFAIL    -1
#define ROTATE_ERUNTIME -2

typedef VISA_Handle ROTATE_Handle;

typedef struct IROTATE_ADAPT_Params ROTATE_Params;

extern ROTATE_Params ROTATE_PARAMS;   /**< Default instance parameters. */

typedef struct IROTATE_ADAPT_InArgs ROTATE_InArgs;


extern SKEL_Fxns ROTATE_SKEL;

extern IROTATE_ADAPT_Fxns ROTATE_STUBS;

/*
 *  ======== ROTATE_create ========
 */
/**
 *  @brief      Create an instance of a color rotation algorithm.
 *
 *  Instance handles must not be concurrently accessed by multiple threads;
 *  each thread must either obtain its own handle (via ROTATE_create()) or
 *  explicitly serialize access to a shared handle.
 *
 *  @param[in]  e       Handle to an opened engine.
 *  @param[in]  name    String identifier of the rotate algorithm
 *                      to create.
 *  @param[in]  params  Creation parameters.
 *
 *  @retval     NULL            An error has occurred.
 *  @retval     non-NULL        The handle to the newly created algorithm
 *                              instance.
 *
 *  @remark     Depending on the configuration of the engine opened, this
 *              call may create a local or remote instance of the algorithm.
 *
 *  @sa         Engine_open()
 */
extern ROTATE_Handle ROTATE_create(Engine_Handle e, String name,
    ROTATE_Params *params);

/*
 *  ======== ROTATE_process ========
 */
/**
 *  @brief      Execute the "process" method in this instance of a ROTATE
 *              algorithm.
 *
 *  @param[in]  handle  Handle to a created ROTATE instance.
 *  @param[in]  inBuf   Pointer to input buffer
 *  @param[out] outBuf  Pointer to output buffer
 *  @param[in]  bufSize Size of input and output buffers
 *  @param[in]  cosine  Cosine of angle used to rotate chroma plane.
 *  @param[in]  sine    Sine of angle used to rotate chroma plane.
 *
 *  @pre        @c handle is a valid (non-NULL) ROTATE handle
 *              and the ROTATE algorithm is in the created state.
 *
 *  @retval     #ROTATE_EOK         Success.
 *  @retval     #ROTATE_EFAIL       Failure.
 *  @retval     #ROTATE_ERUNTIME    Internal Runtime Error.
 *
 *  @remark     This is a blocking call, and will return after the data
 *              has been decoded.
 *
 *  @remark     The buffers supplied to ROTATE_process() may have constraints
 *              put on them.  For example, in dual-processor, shared memory
 *              architectures, where the codec is running on a remote
 *              processor, the buffers may need to be physically contiguous.
 *              Additionally, the remote processor may place restrictions on
 *              buffer alignment.
 *
 *  @sa         ROTATE_create(), ROTATE_delete()
 */
extern Int32 ROTATE_process(ROTATE_Handle handle, char inBuf[],
    char outBuf[], XDAS_Int32 bufSize, XDAS_Int16 cosine, XDAS_Int16 sine);


/*
 *  ======== ROTATE_delete ========
 */
/**
 *  @brief      Delete the instance of a color rotation algorithm.
 *
 *  @param[in]  handle  Handle to a created algorithm instance.
 *
 *  @remark     Depending on the configuration of the engine opened, this
 *              call may delete a local or remote instance of the algorithm
 *
 *  @pre        @c handle is a valid (non-NULL) handle which is
 *              in the created state.
 *
 *  @post       All resources allocated as part of the ROTATE_create()
 *              operation (memory, DMA channels, etc.) are freed.
 *
 *  @sa         ROTATE_create()
 */
extern Void ROTATE_delete(ROTATE_Handle handle);

#ifdef __cplusplus
}
#endif

#endif

