//******************************************************************************
// Complex 32-bit FFT power benchmark
//
//! \example main.c
//! This example demonstrates the power consumption of the LEA IP itself. If P1.0
//! is driven high then LEA will execute a 32 bit Complex FFT. If P1.0 is not
//! driven high, then LEA will poll the LEA IFG. Two power measurments will be
//! taken, one for when LEA is executing and the other for when LEA IFG is being
//! polled. Subtract (CPU+LEA) - CPU = LEA power consumption where:
//! CPU+LEA = P1.0 being driven high
//! CPU = P1.0 being driven low
//!
// Evan Wakefield
// Texas Instruments Inc.
// August 2016
//******************************************************************************
#include "msp430.h"

#include <math.h>
#include <stdint.h>
#include <stdbool.h>

#include "DSPLib.h"

/* Input signal parameters */
#define FS                  8192
#define SAMPLES             256
#define SIGNAL_FREQUENCY1   200
#define SIGNAL_AMPLITUDE1   0.6
#define SIGNAL_FREQUENCY2   2100
#define SIGNAL_AMPLITUDE2   0.15

/* Constants */
#define PI                  3.1415926536

#define CLOCK_16MHZ         1
#define CLOCK_8MHZ          0

#define ZEROS_INPUT         0
#define SIN_INPUT           0
#define A5_INPUT            1

// Flags
volatile bool runLEA;

/* Input vector 1 */
DSPLIB_DATA(input,MSP_ALIGN_CMPLX_FFT_IQ31(SAMPLES))
_iq31 input[SAMPLES*2];

/* Temporary data array for processing */
DSPLIB_DATA(temp,4)
_q15 temp[SAMPLES];

void init_memory(uint32_t * pointer, short int size, uint32_t value);
void CS_init(void);
void initSignal(void);
void initHamming(void);

msp_cmplx_fft_iq31_params fftParams;

/* Benchmark cycle counts */
volatile uint32_t cycleCount;

void main(void)
{


    // stop WDT
    WDTCTL = WDTPW+WDTHOLD;

    CS_init();

    // Configure ports for low power
    PADIR = 0;  PAOUT = 0;  PAREN = 0xFFFF;
    PBDIR = 0;  PBOUT = 0;  PBREN = 0xFFFF;
    PCDIR = 0;  PCOUT = 0;  PCREN = 0xFFFF;
    PDDIR = 0;  PDOUT = 0;  PDREN = 0xFFFF;
    PJDIR = 0;  PJOUT = 0;  PJREN = 0xFFFF;

    // unlock port
    PMMCTL0_H = PMMPW_H;
    PM5CTL0_L &= ~(LOCKLPM5_L);                 // clear lock bit


    initSignal();

    /* Initialize the fft parameter structure. */
    fftParams.length = SAMPLES;
    fftParams.bitReverse = 1;
    fftParams.twiddleTable = msp_cmplx_twiddle_table_256_q15;

    // if P1.0 is driven high, then LEA will execute, otherwise, we will run without LEA
    if((P1IN & BIT0) == 1) {
        runLEA = true;
    }
    else {
        runLEA = false;
    }

    while(1) {
        if(runLEA){
            msp_cmplx_fft_iq31(&fftParams, input);
        }
        else{
            while(!msp_lea_ifg);
        }
    }
}

//! To initalize memory locations with selected value
void init_memory(uint32_t * pointer, short int size, uint32_t value){
    short int i = 0;
    while(i<size)
    {
        pointer[i] = value;
        i++;
    }
}

void CS_init(){
#if CLOCK_16MHZ
    // Configure one FRAM waitstate as required by the device datasheet for MCLK
    // operation beyond 8MHz _before_ configuring the clock system.
    FRCTL0 = FRCTLPW | NWAITS_1;

    // Clock System Setup
    CSCTL0_H = CSKEY_H;                     // Unlock CS registers
    CSCTL1 = DCOFSEL_0;                     // Set DCO to 1MHz

    // Set SMCLK = MCLK = DCO, ACLK = VLOCLK
    CSCTL2 = SELA__VLOCLK | SELS__DCOCLK | SELM__DCOCLK;

    // Per Errata CS12 set divider to 4 before changing frequency to
    // prevent out of spec operation from overshoot transient
    CSCTL3 = DIVA__4 | DIVS__4 | DIVM__4;   // Set all corresponding clk sources to divide by 4 for errata
    CSCTL1 = DCOFSEL_4 | DCORSEL;           // Set DCO to 16MHz

    // Delay by ~10us to let DCO settle. 60 cycles = 20 cycles buffer + (10us / (1/4MHz))
    __delay_cycles(60);

    CSCTL3 = DIVA__1 | DIVS__16 | DIVM__1;   // Set MCLK to 16MHz, SMCLK to 1MHz

    CSCTL4 = HFXTOFF + LFXTOFF;             // Cancel external clocks
    CSCTL5 &= ~(HFXTOFFG + LFXTOFFG);

    CSCTL0_H = 0;                           // Lock CS Registers
#endif

#if CLOCK_8MHZ

    //    //  Clock System Configuration
    //    CSCTL0_H = 0xA5;                              // Unlock CS
    //    CSCTL1 = 0x000C;                            // set standard settings for silicon (DCORSEL=0x0,DCOFSEL=0x6 resulting in 8MHz DCOCLK)
    //    CSCTL2 = SELA_1 + SELS_3 + SELM_3;          // set ACLK = VLOCLK, MCLK/SMCLK = DCO
    //    CSCTL3 = DIVS_0 + DIVM_0;                 // set MCLK (IFCLK divider) 8MHz

    //Ensure that there are 0 wait states enabled
    FRCTL0 = FRCTLPW | NWAITS_0;


    // Clock System Setup
    CSCTL0_H = CSKEY_H;                     // Unlock CS registers
    CSCTL1 = DCOFSEL_0;                     // Set DCO to 1MHz

    // Set SMCLK = MCLK = DCO, ACLK = VLOCLK
    CSCTL2 = SELA__VLOCLK | SELS__DCOCLK | SELM__DCOCLK;

    // Per Errata CS12 set divider to 4 before changing frequency to
    // prevent out of spec operation from overshoot transient
    CSCTL3 = DIVA__4 | DIVS__4 | DIVM__4;   // Set all corresponding clk sources to divide by 4 for errata
    CSCTL1 = DCOFSEL_6;                     // Set DCO to 8MHz

    // Delay by ~10us to let DCO settle. 60 cycles = 20 cycles buffer + (10us / (1/4MHz))
    __delay_cycles(60);

    CSCTL3 = DIVA__1 | DIVS__8 | DIVM__1;   // Set MCLK  to 8MHz, SMCLK to 1MHz

    CSCTL4 = HFXTOFF + LFXTOFF;             // Cancel external clocks
    CSCTL5 &= ~(HFXTOFFG + LFXTOFFG);

    CSCTL0_H = 0;                           // Lock CS Registers
#endif
}

void initSignal(void)
{
#if SIN_INPUT
    uint16_t i;
    msp_status status;
    msp_cmplx_q15_params cmplxParams;
    msp_sinusoid_q15_params sinParams;
    msp_shift_iq31_params shiftParams;
    msp_q15_to_iq31_params convertParams;

    /* Generate Q15 input signal 1 */
    sinParams.length = SAMPLES;
    sinParams.amplitude = _Q15(SIGNAL_AMPLITUDE1);
    sinParams.cosOmega = _Q15(cosf(2*PI*SIGNAL_FREQUENCY1/FS));
    sinParams.sinOmega = _Q15(sinf(2*PI*SIGNAL_FREQUENCY1/FS));
    status = msp_sinusoid_q15(&sinParams, temp);
    msp_checkStatus(status);

    /* Generate Q15 input signal 2 to temporary array */
    sinParams.length = SAMPLES;
    sinParams.amplitude = _Q15(SIGNAL_AMPLITUDE2);
    sinParams.cosOmega = _Q15(cosf(2*PI*SIGNAL_FREQUENCY2/FS));
    sinParams.sinOmega = _Q15(sinf(2*PI*SIGNAL_FREQUENCY2/FS));
    status = msp_sinusoid_q15(&sinParams, &temp[SAMPLES]);
    msp_checkStatus(status);

    /* Combine signal 1 and 2 as real and imaginary respectively. */
    for (i = 0; i < SAMPLES; i++) {
        input[i*2+0] = (_iq31)temp[i];
        input[i*2+1] = (_iq31)temp[i+SAMPLES];
    }
#endif

#if ZEROS_INPUT
    init_memory(input, SAMPLES*2, 0x0);
#endif

#if A5_INPUT
    init_memory(input, SAMPLES*2, 0xA5A5A5A5);
#endif
}

