/* --COPYRIGHT--,BSD
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
// *******************************************************************************
//	Example code demonstrating oversampling to get higher resolution
//
//	DAC8560 EVM jumpers:
//	W1 position +3VA	W2 position 1
//	W3 position open	W4 position open
//	W5 position 3		W6 position 1
//	W7 position open	W8 position open
//	W15 position open	J6 position 5 tied to 6
//
//  MSP-EXP430FR4133	DAC8560 EVM:
//	3V3					J6 1 (VCC)
//	5.2	UCB0SIMO		J2 11 (SDI)
//	5.1 UCB0CLK			J2 3 (SCLK)
//	5.0 UCB0STE			J2 1 (CS)
//	1.1 VeREF+			W4 2 (VREF)
//	1.3	A3				J4 2 (VOUT)
//	GND					J2 18 (GND)
//	3V3					J6 8 (+3.3VA)
//
//  R Brown
//  Texas Instruments Inc.
//  December 2015
//  Built with CCS Version: 6.0.1
//******************************************************************************

#include <msp430.h>
#include <math.h>

unsigned long uncalibrated_result = 0;
unsigned long result = 0;
unsigned long ADCresult;                    // Result after oversampling & averaging
                                            // 16 samples for 12-bit ADC

unsigned int i = 0;
const int extrabits = 2;					// Extra bits of resolution, controls the oversampling factor
const int period = 4;						// 256 [over]samples per second (ACLK/(2*period*oversampling factor))
unsigned int data = 0x00;					// Data to be sent out
unsigned int oversampling = 0;				// Oversampling factor, set at the start of main

unsigned long movingAVG = 0;
#pragma PERSISTENT(AVGBuffer)				// Buffer must be placed in FRAM if too large for RAM space
unsigned int AVGBuffer [16] = {0};			// Buffer is the same size as the oversampling factor
unsigned int AVGpnt = 0;

void sendline(unsigned int, unsigned long);

#define CAL_ADC_GAIN_FACTOR  *((unsigned int *)(0x1A14 + 0x0002))   	// ADC gain factor
#define CAL_ADC_OFFSET  *((unsigned int *)(0x1A14 + 0x0004))   			// ADC offset

int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;                   // Stop WDT

    oversampling = pow(4, extrabits);			// Oversampling factor

    SYSCFG0 &= ~PFWP;
    for (i = 0; i < oversampling; i++) AVGBuffer[i] = 0;
    SYSCFG0 |= PFWP;

    // Set P1.0 as UCA0 UART pins and P1.1 as Veref+ pin
    // Initialize P1.2 for push button
    P1SEL0 |= BIT0 | BIT1;
    P1REN |= BIT2;
    P1OUT |= BIT2;

    // Set P5.0 to P5.3 as UCB0 SPI pins
    P5SEL0|= BIT0 + BIT1 + BIT2 + BIT3;

    // Configure ADC A3 pin
    SYSCFG2 |= ADCPCTL3;

    // Configure XT1 oscillator
    P4SEL0 |= BIT1 | BIT2;                      // P4.1 & P4.2: crystal pins

    // Disable the GPIO power-on default high-impedance mode to activate
    // previously configured port settings
    PM5CTL0 &= ~LOCKLPM5;

    __bis_SR_register(SCG0);                	// Disable FLL
    CSCTL3 |= SELREF__REFOCLK;              	// Set REFO as FLL reference source
    CSCTL0 = 0;                             	// clear DCO and MOD registers
    CSCTL1 &= ~(DCORSEL_7);                 	// Clear DCO frequency select bits first
    CSCTL1 |= DCORSEL_3;                    	// Set DCO = 1MHz
    CSCTL2 = FLLD_0 + 243;                  	// DCODIV = 1MHz
    __delay_cycles(3);
    __bic_SR_register(SCG0);                	// enable FLL
    while(CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1)); 	// Poll until FLL is locked

    CSCTL4 = SELA__XT1CLK;                      // Set ACLK = XT1; MCLK = SMCLK = DCO
    do
    {
        CSCTL7 &= ~(XT1OFFG | DCOFFG);          // Clear XT1 and DCO fault flag
        SFRIFG1 &= ~OFIFG;
    }while (SFRIFG1 & OFIFG);                   // Test oscillator fault flag

    //Initialize SPI
    UCB0CTLW0 |= UCSWRST;
    UCB0CTLW0 |= UCMST | UCSYNC | UCCKPL | UCMSB | UCMODE_2 | UCSTEM;
    UCB0CTLW0 |= UCCKPH | UCSSEL__MODCLK;
    UCB0BR0 = 0x0F;
    UCB0BR1 = 0;
    UCB0CTLW0 &= ~UCSWRST;

    //Initialize UART
    UCA0CTLW0 |= UCSWRST;
    UCA0CTLW0 |= UCSSEL__SMCLK;

    // Baud Rate calculation
    // 1000000/(16*9600) = 6.51
    // Fractional portion = 0.51
    // User's Guide Table 14-4: UCBRSx = 0x20
    // UCBRFx = int ( (6.51-6)*16) = 8
    UCA0BR0 = 52;                             	// 1000000/16/9600
    UCA0BR1 = 0x00;
    UCA0MCTLW = 0x4900 | UCOS16 | UCBRF_1;

    UCA0CTLW0 &= ~UCSWRST;                    	// Initialize eUSCI

    // Configure ADC;
    ADCCTL0 = ADCSHT_4 | ADCON;                	// 64 cycles, ADCON
    ADCCTL1 = ADCSHP | ADCSHS_2 | ADCCONSEQ_2;  // Rpt single ch; TA1.1 trigger
    ADCCTL2 = ADCRES + ADCSR;                   // 10-bit conversion results, buffer sampling rate to 50 ksps
    ADCMCTL0 = ADCINCH_3 | ADCSREF_3;           // Veref+ 2.5v, A3
    ADCIE |= ADCIE0;               				// Enable ADC conv complete interrupt

    // Configure reference
    PMMCTL0_H = PMMPW_H;                        // Unlock the PMM registers
    PMMCTL2 |= INTREFEN;                        // Enable internal reference
    __delay_cycles(400);                        // Delay for reference settling

    // Configure ADC timer trigger TA1.1
    TA1CCR0 = period;                           // PWM Period
    TA1CCR1 = period/2;                         // Duty cycle TA1.1
    TA1CCTL1 = OUTMOD_4;                        // TA1CCR1 toggle
    TA1CTL = TASSEL_1 | MC_3 | TACLR;           // ACLK, up/down mode
    ADCCTL0 |= ADCENC;                          // Enable conversion

    while(P1IN & BIT2);							// Wait for P1.2 button press

    while(1)
    {
        __bis_SR_register(LPM3_bits + GIE);     // LPM3, TA0_ISR will force exit
        __no_operation();
        ADCresult = movingAVG >> extrabits;

        data += pow(2, 6-extrabits);

        while (!(UCB0IFG&UCTXIFG));
        UCB0TXBUF = 0x00;

        while (!(UCB0IFG&UCTXIFG));
        UCB0TXBUF = data>>8;

        while (!(UCB0IFG&UCTXIFG));
        UCB0TXBUF = data;

        sendline(data-pow(2, 6-extrabits), ADCresult);

        if (data >= 0xFFF0){data = 0; while(1) __no_operation();}

    }
}

// ADC interrupt service routine
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=ADC_VECTOR
__interrupt void ADC_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(ADC_VECTOR))) ADC_ISR (void)
#else
#error Compiler not supported!
#endif
{
    switch(__even_in_range(ADCIV,ADCIV_ADCIFG))
    {
        case ADCIV_NONE:
            break;
        case ADCIV_ADCOVIFG:
            break;
        case ADCIV_ADCTOVIFG:
            break;
        case ADCIV_ADCHIIFG:
            break;
        case ADCIV_ADCLOIFG:
            break;
        case ADCIV_ADCINIFG:
            break;
        case ADCIV_ADCIFG:
            //Automatically clears ADC10IFG0 by reading memory buffer
            uncalibrated_result = ADCMEM0;
            result = (uncalibrated_result*CAL_ADC_GAIN_FACTOR/32768)+CAL_ADC_OFFSET;

            SYSCFG0 &= ~PFWP;
            movingAVG -= AVGBuffer[AVGpnt];
            movingAVG += result;
            AVGBuffer[AVGpnt++] = result;
            SYSCFG0 |= PFWP;
            if (AVGpnt == oversampling)
            {
              AVGpnt = 0;
              //Clear CPUOFF bit from 0(SR)
              //Breakpoint here and watch ADC_Result
              __bic_SR_register_on_exit(LPM3_bits);
            }

            break;
        default:
            break;
    }
}

void sendline(unsigned int step, unsigned long voltage)
{
	unsigned int i = 0;
	const char stepstring[] = {"Step: "};				//Strings being sent to the PC
	const int stepsize = 6;								//String size
	const char voltagestring[] = {"Voltage: "};
	const int voltagesize = 9;
	const char endlinestring[] = {"\n\r"};				//Command to end line
	const int endlinesize = 4;

	//Send the step string
	for(i=0; i<=stepsize; i++)
	{
		while(!(UCA0IFG&UCTXIFG));
		UCA0TXBUF = stepstring[i];
	}

	//Sends the step in hexadecimal
	while(!(UCA0IFG&UCTXIFG));
	if(((step>>12)&0xF)>=0x0A) UCA0TXBUF = ((step>>12)&0xF) + 0x37;
	else UCA0TXBUF = ((step>>12)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((step>>8)&0xF)>=0x0A) UCA0TXBUF = ((step>>8)&0xF) + 0x37;
	else UCA0TXBUF = ((step>>8)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((step>>4)&0xF)>=0x0A) UCA0TXBUF = ((step>>4)&0xF) + 0x37;
	else UCA0TXBUF = ((step>>4)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if((step&0xF)>=0x0A) UCA0TXBUF = (step&0xF) + 0x37;
	else UCA0TXBUF = (step&0xF) + 0x30;

	//Insert a tab to separate the step from the voltage
	while(!(UCA0IFG&UCTXIFG));
	UCA0TXBUF = 0x09;

	//Send the voltage string
	for(i=0; i<=voltagesize; i++)
	{
		while(!(UCA0IFG&UCTXIFG));
		UCA0TXBUF = voltagestring[i];
	}

	//Sends the voltage in hexadecimal
	while(!(UCA0IFG&UCTXIFG));
	if(((voltage>>12)&0xF)>=0x0A) UCA0TXBUF = ((voltage>>12)&0xF) + 0x37;
	else UCA0TXBUF = ((voltage>>12)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((voltage>>8)&0xF)>=0x0A) UCA0TXBUF = ((voltage>>8)&0xF) + 0x37;
	else UCA0TXBUF = ((voltage>>8)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((voltage>>4)&0xF)>=0x0A) UCA0TXBUF = ((voltage>>4)&0xF) + 0x37;
	else UCA0TXBUF = ((voltage>>4)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if((voltage&0xF)>=0x0A) UCA0TXBUF = (voltage&0xF) + 0x37;
	else UCA0TXBUF = (voltage&0xF) + 0x30;

	//Start a new line for the next string
	for(i=0; i<=endlinesize; i++)
	{
		while(!(UCA0IFG&UCTXIFG));
		UCA0TXBUF = endlinestring[i];
	}
}
