/*******************************************************************************
 *  CTS_SYSBIOS_Interface.c
 *
 *  Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/
/***************************************************************************//**
 * @file   CTS_SYSBIOS_Layer.c
 *
 * @brief
 *
 * @par    Project:
 *              MSP430 SYSBIOS Capacitive Touch Interface
 *
 * @par    Developed using:
 *              CCS Version : 5.1.0.09000, w/support for GCC extensions (--gcc)
 *              RTSC/XDCtools: 3.22.04.46
 *              SYS/BIOS: 6.33.04.39
 *
 * @version     1.0.0 Initial Release
 *
 * @par    Clock Functions(Swis):
 *
 *              - TI_CTS_SYSBIOS_Measure_Sensor()
 * 				- TI_CTS_SYSBIOS_Baseline_Init()
 *
 * @par    Swis:
 *
 *              - TI_CTS_SYSBIOS_Measure_Element()
 *              - TI_CTS_SYSBIOS_Deglitch_Sensor()
 *              - TI_CTS_SYSBIOS_Deglitch_Element()
 *
 * @par    Tasks:
 *              - TI_CTS_SYSBIOS_Task()
 *
 * @par    Hwi:
 *              - TI_CTS_SYSBIOS_WDTa_Interrupt()
 *
 * @par    Non-SYSBIOS:
 *
 *              - TI_CTS_Copy_Flash_to_RAM()
 *
 ******************************************************************************/

// NOTE: COMPILER BUG WORKAROUND-ALWAYS PLACE <xdc/cfg/global.h>
// before <msp430.h>
#include <xdc/cfg/global.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/Clock.h>
#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/family/msp430/Power.h>

// with active links for software configuration management use include feature to point to GIT directory
//#include <CTS/CTS_SYSBIOS_Interface.h>
//#include <CTS/CTS_SYSBIOS_Layer.h>

#include <CTS_SYSBIOS_Interface.h>
#include <CTS_SYSBIOS_Layer.h>

RamSensorConfig* currentSensor;
uint16_t elementIndex,sensorIndex,deglitch;
uint8_t ctsMode;

/***************************************************************************//**
 * @addtogroup CTS_SYSBIOS_Interface
 * @{
 ******************************************************************************/

/***************************************************************************//**
 * @brief   Write information from flash into RAM and 'initialize' the device
 *
 *          The critical feature of this function is the establishment of
 *          both the baseline and the filter based upon a previously stored
 *          value which represents the untouched capacitance of the system.
 *
 * @param   none
 * @return  none
 ******************************************************************************/
void TI_CTS_Copy_Flash_to_RAM(void)
{
	volatile uint8_t i,j;

    // Init
	for(i=0;i<TOTAL_NUMBER_OF_ELEMENTS;i++)
	{
		eConfig[i].portInfo = &eport[i];
		eConfig[i].eRegister0 = eConfigFlash[i].eRegister0;
		eConfig[i].threshold = eConfigFlash[i].threshold;
		eConfig[i].baseline = eConfigFlash[i].baseline;
		eConfig[i].maxResponse = eConfigFlash[i].maxResponse;
        for(j=0;j<FILTER_SIZE;j++)
        {
        	filter[i][j] = eConfig[i].baseline;
        }
	}
	for(i=0;i<TOTAL_NUMBER_OF_SENSORS;i++)
	{
		sConfig[i].sRegister0 = sConfigFlash[i].sRegister0;
		sConfig[i].sRegister1 = sConfigFlash[i].sRegister1;
	}
}

/***************************************************************************//**
 * @brief   Start the sequence for measuring a sensor.  Measure first element.
 *
 *          This function is called by a Clock Swi in SYS/BIOS. The sensorIndex
 *          identifies which sensor in the array sConfig[] is measured.
 *
 * @param   sensorIndex Index for the sensor to be measured
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Measure_Sensor(uintptr_t sensorIndex)
{
	uint16_t eMask;

    elementIndex = 0;
    ctsMode = MEASURE;
    // Identify which sensor is being measured
    currentSensor = &sConfig[sensorIndex];
	eMask = BIT0;
	while(!(currentSensor->sRegister0.elementLocation & eMask)
			&& (elementIndex < TOTAL_NUMBER_OF_ELEMENTS))
	{
		//find elements within sensor
		elementIndex++;
		eMask = eMask << 1;
	}
	TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL(&eConfig[elementIndex]);
}

/***************************************************************************//**
 * @brief   Start the sequence for initializing the baseline for all elements
 *          within a sensor.
 *
 *          This function is optional and can be used in lieu of the baseline
 *          established by TI_CTS_Copy_Flash_to_RAM().The sensorIndex
 *          identifies which sensor in the array sConfig[] is initialized.
 *
 * @param   sensorIndex Index for the sensor to be updated.
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Baseline_Init(uintptr_t sensorIndex)
{
    uint16_t eMask;

    elementIndex = 0;
    ctsMode = BASE_INIT;
    // Identify which sensor is being measured
    currentSensor = &sConfig[sensorIndex];
	eMask = BIT0;
	while(!(currentSensor->sRegister0.elementLocation & eMask) && (elementIndex < TOTAL_NUMBER_OF_ELEMENTS))
	{
		//find elements within sensor
		elementIndex++;
		eMask = eMask << 1;
	}
	TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL(&eConfig[elementIndex]);
}

/***************************************************************************//**
 * @brief   Measure remaining elements in sensor.
 *
 *			This function updates the elementIndex and measures all of the
 *			elements within the sensor.
 *
 * @param   arg
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Measure_Element(uintptr_t arg)
{
	uint16_t eMask;

	elementIndex++;
	eMask = BIT0;
	eMask = eMask << elementIndex;
	while(!(currentSensor->sRegister0.elementLocation & eMask) && (elementIndex < TOTAL_NUMBER_OF_ELEMENTS))
	{
		//find elements within sensor
		elementIndex++;
		eMask = eMask << 1;
	}
	// Initial measurement of sensor
	if(elementIndex == TOTAL_NUMBER_OF_ELEMENTS)
	{
		if(ctsMode == MEASURE)
		{
			sensorIndex = 0;
		    Swi_post(deglitchSensor);
		}
	}
	else
	{
		TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL(&eConfig[elementIndex]);
	}
}

/***************************************************************************//**
 * @brief   Deglitch each element within sensor that registered a detection in
 * 			the initial measurement.
 *
 *          This function updates the appropriate indices and determines if the
 *          deglitch criteria has been met for each element.  If met then the
 *          appropriate flags are set in the sensor result register.
 *
 * @param   arg
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Deglitch_Element(uintptr_t arg)
{
	uint16_t eMask, sMask;

	eMask = BIT0;
	sMask = BIT0;
	eMask = eMask << elementIndex;
	sMask = sMask << sensorIndex;
	while(!(currentSensor->sRegister0.elementLocation & eMask)
			&& (elementIndex < TOTAL_NUMBER_OF_ELEMENTS))
	{
		//find elements within sensor
		elementIndex++;
		eMask = eMask << 1;
	}
	// Now that the initial data collection has been completed identify
	// if deglith is needed.
	// detection
	// Determine if detection occurred
	if(sensorIndex == currentSensor->sRegister0.numberOfElements)
	{
		// all of the keys have been deglitched
		Semaphore_post(ctsSem);
	}
	else
	{
		if((eConfig[elementIndex].measure
			< (eConfig[elementIndex].baseline-eConfig[elementIndex].threshold))
			&& (eConfig[elementIndex].baseline>eConfig[elementIndex].threshold))
		{
			if(deglitch == 0)
			{
				currentSensor->sRegister2.detail |= sMask;
				currentSensor->sRegister2.detect = 1;
				//move the index startover deglitch
				Swi_post(deglitchSensor);
			}
			else
			{
				deglitch --;
				// remeasure
				TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL(&eConfig[elementIndex]);
			}
		}
		else
		{
			currentSensor->sRegister2.detail &= ~sMask;
			//move the index startover deglitch
			Swi_post(deglitchSensor);
		}
	}
}

/***************************************************************************//**
 * @brief   Initialize or increment variables to deglitch elements in Sensor.
 *
 * @param   arg
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Deglitch_Sensor(uintptr_t arg)
{
	if(ctsMode == MEASURE)
	{
		ctsMode = DEGLITCH;
		elementIndex = 0;
		currentSensor->sRegister2.detect = 0;
	}
	else
	{
		elementIndex++;
		sensorIndex++;
	}
	deglitch = currentSensor->sRegister1.deglitch;
	Swi_post(deglitchElement);
}

/***************************************************************************//**
 * @brief   Perform post-processing once measurement of sensor is complete.
 *
 *          After each element within a sensor is measured the information
 *          is post processed in a non time-critical task.
 *
 * @param   arg
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Task(void)
{
    // Do this forever
    while (1)
    {
    	Semaphore_pend(ctsSem, BIOS_WAIT_FOREVER);
    	TI_CTS_SYSBIOS_Post_Process(&sConfig[0]);
    }
}

/***************************************************************************//**
 * @brief   WDTA ISR
 *
 *          The WDTA is handled and the measurement information in timerA1 is
 *          recorded for the element.
 *
 * @param   arg
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_WDTa_Interrupt(void)
{
	volatile uint8_t i;
	TA1CCTL1 ^= CCIS0;                   // Create SW capture of CCR1
    CBCTL1 &= ~CBON;                     // Turn off comparator
	// record value
	WDTCTL = (WDTPW + WDTHOLD);
	SFRIE1 &= ~WDTIE;
	if(ctsMode == MEASURE)
	{
		eConfig[elementIndex].measure = TA1CCR1;          // Save result
		Swi_post(measureElement);
	}
	else if(ctsMode == DEGLITCH)
	{
		eConfig[elementIndex].measure = TA1CCR1;          // Save result
		Swi_post(deglitchElement);
	}
	else if(ctsMode == BASE_INIT)
	{
		eConfig[elementIndex].baseline = TA1CCR1;          // Save result
		for(i=0;i< FILTER_SIZE;i++)
		{
		    filter[elementIndex][i] = TA1CCR1;
		}
		Swi_post(measureElement);
	}
}

/***************************************************************************//**
 * @}
 ******************************************************************************/
