/*******************************************************************************
 *  CTS_SYSBIOS_HAL.c
 *
 *  Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/
/*******************************************************************************
 * @file CTS_SYSBIOS_HAL.c
 *
 * @brief  Hardware abstraction of various combinations of modules to
 *         perform a capacitance measurement.
 *
 * @par    Project:
 *              MSP430 SYSBIOS Capacitive Touch Interface
 *
 * @par    Developed using:
 *              CCS Version : 5.1.0.09000, w/support for GCC extensions (--gcc)
 *              RTSC/XDCtools: 3.22.04.46
 *              SYS/BIOS: 6.33.04.39
 *
 * @version     1.0.0 Initial Release
 *
 * @par    Supported Hardware Configurations:
 *              - TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL()
 * 
 ******************************************************************************/


/***************************************************************************//**
 * @addtogroup CTS_SYSBIOS_HAL
 * @{
 ******************************************************************************/

#include "CTS_SYSBIOS_HAL.h"

#ifdef RO_COMPB_TA1_WDTA
/***************************************************************************//**
 * @brief   RO method capacitance measurement using CompB, TimerA1, and WDTA
 *
 *          Schematic Description of CompB forming relaxation oscillator and
 *          coupling (connection) between the relaxation oscillator and TimerA1.
 *          <- Output
 *          -> Input
 *          R  Resistor (typically 100Kohms)
 * 
 *          element-+-R----<-CBOUT/TA1CLK
 *                  |
 *                  +-------> CBx
 * 
 *          The WDTA interval represents the measurement window.  The number of 
 *          counts within the TA1R that have accumulated during the measurement
 *          window represents the capacitance of the element.
 * 
 * @param   group Address of the structure describing the Sensor to be measured
 * @param   counts Address to where the measurements are to be written
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL(RamElementConfig *element)
{
    //** Setup Measurement timer***************************************************
    // connect CBOUT with TA1
    *(compbTaxConfig.cboutTAxDirRegister) |= (compbTaxConfig.cboutTAxBits);
    *(compbTaxConfig.cboutTAxSelRegister) |= (compbTaxConfig.cboutTAxBits);
    // Setup Comparator
    // Vcc = 3.3V
    //       Vrefh = 3.3*(24+1)/32 = 2.58V
    //       Vrefl = 3.3*(4+1)/32 = 0.52V
    CBCTL2 = CBREF14+CBREF13 + CBREF02;

    // Configure Timer TA1
    TA1CTL = TASSEL_0+MC_2;                   // TACLK, cont mode
    TA1CCTL1 = CM_3+CCIS_2+CAP;               // Pos&Neg,GND,Cap
    // Turn on Comparator
    CBCTL1 = CBON;                            // Turn on comparator without filter
    CBCTL3 |= (compbTaxConfig.cbpdBits);      // set CPD bits to disable
    SFRIE1 |= WDTIE;                          // enable WDT interrupt
    CBCTL2 |= CBRS_1;                         // Turn on reference

    CBCTL0 = CBIMEN + element->portInfo->inputBits;
    //**  Setup Gate Timer ********************************************************
    // Set duration of sensor measurement from element configuration Register0.
    WDTCTL = WDTPW+WDTTMSEL+WDTCNTCL+ element->eRegister0.measureTime;
    TA1CTL |= TACLR;                          // Clear Timer_A TAR
}

#endif
/***************************************************************************//**
 * @}
 ******************************************************************************/

