/******************************************************************************
;   Robust Glass Breakage Detector "Main C file"
;
;   B. Nisarga/ K. Venkat
;   Texas Instruments Inc.
;   December 2007
;   Built with IAR Embedded Workbench Version: 4.10A
;*******************************************************************************/

#include "msp430x22x4.h"
#include <intrinsics.h>

#define TACCR0_2ms  25                      // TACCR0 = 20 @ VLO=12khz
#define HIGH_LVL    100                     // threshold to detect ACTIVITY
#define LOW_LVL     -40                     // threshold to detect ACTIVITY 
#define THUD_LVL    35                      // threshold to confirm valid THUD

// Define and initialize global variables
volatile unsigned int i, discard_count=0, sample_count;
volatile int temp1;
int output1;
long average=0, sum =0;
unsigned char discard_flag=1, activity=0, OA1_delay_ON=0;

// Function prototypes
int wdf_thud(int); 
void signal_analysis_one(void);
void signal_analysis_two(void);
void init_glass_break(void);
void INIT_wdf_2k(void);
void delay_8MHz_no_AAF(void);
void INITMEM(void);
void INIT_WDF(void);

void main(void)
{
    WDTCTL = WDTPW + WDTHOLD;               // Stop watchdog timer
    
    P1DIR= 0xFF;                            // Configure Port pins
    P1OUT= 0;
    P2DIR= 0xFF; 
    P2OUT= 0;
    P3DIR= 0xFF;
    P3OUT= 0;
    P4DIR= 0xFF;
    P4OUT= 0;
                                            // Configure Clock system
    BCSCTL1= CALBC1_12MHZ;                  // Set DCO = 12MHz
    DCOCTL = CALDCO_12MHZ;     
    BCSCTL3 = LFXT1S_2;                     // ACLK = VLO
    
                                            // Configure ADC10
    ADC10CTL0 = ADC10SHT_1+ADC10SR;         // Sample and hold time = 8xADC10CLKS
    ADC10CTL1= INCH_1+ADC10DIV_2+ADC10SSEL_3+CONSEQ_0; // input channel A1,
                                            // ADC10CLK=SMCLK/3, single-channel-
                                            // single-conversion mode    
    
                                            // Enable Op-amp 0
    ADC10AE0 = BIT0+BIT1+BIT2;              // A0+A1+A2 analog enable (OA0 i/ps and o/p) 
    OA0CTL1= OAFBR_6+OAFC_6+OANEXT;         // Inv PGA, Gain = -7, ext inv input
    OA0CTL0= OAN_1+OAADC1;                  // Output high Z, OA0 output to 
                                            // ADC10 input channel A1 
  
                                            // Enable Op-amp 1
    ADC10AE1 = 0x0A0;                       // A13+A15 analog enable (OA1 +ve i/p and o/p)
    OA1CTL1= OAFC_2;                        // Unity gain buffer mode
    OA1CTL0= OAP_3+OAADC0;                  // Output high Z, OA0 output to   
                                            // ADC10 input channel A13  
  
                                            // Configure Timer_A to generate 2.5ms device wake-up
    TACTL= TASSEL0;                         // TACLK = ACLK=VLO 
    TACCTL0 = CCIE;                         // Capture compare interrupt enable  
    TACCR0 = TACCR0_2ms;                    // Load TACCR0 with 2.5ms count
    TACTL |= MC_1;                          // Up mode, TA start
    
    while(1)
    {
        if(activity)                        // Check actvity flag
            __bis_SR_register(LPM0_bits + GIE);  // LPM0,  Global Interrupts Enable
        else
            __bis_SR_register(LPM3_bits + GIE);  // LPM3, Global Interrupts Enable
            
            P4OUT |=BIT0;                   // Microphone ON
            P2OUT |=BIT4;                   // Offset Voltage ON     
            OA0CTL0 |=OAPM_3;               // Fast mode for Gain Op-amp 
            delay_8MHz_no_AAF();            // Delay for gain opamp

            ADC10CTL0 |= ADC10ON;
            ADC10CTL0 |= ENC + ADC10SC;     // Enable ADC10 + start conversion 
            while((ADC10BUSY & ADC10CTL0)); // Check for conversion complete
            OA0CTL0 &= ~(OAPM_3);           // Gain-Op-amp OFF
            P4OUT &= ~BIT0;                 // Microphone OFF
            P2OUT &= ~BIT4;                 // Voltage offset OFF                 
            temp1= ADC10MEM-500;            // Conv result - op-amp ref voltage
            if(discard_flag)                // Discard first 200 samples
            {
                if(++discard_count > 200)
                    discard_flag = 0;
                ADC10CTL0 &= ~ENC;          
                ADC10CTL0 &= ~ADC10ON;      // Power-off ADC10  
                TACCTL0 = CCIE; 
                TACCR0 = TACCR0_2ms;
                TACTL |= MC_1;              // Re-enable Timer, Up mode    
            }
            else                            // Valid if discard_flag = 0
            {
                // Compare sampled data with thresholds to detect activity
                if((temp1 >= HIGH_LVL) || (temp1 <= LOW_LVL))
                {
                    // ACTIVITY Detected
                    BCSCTL1= CALBC1_8MHZ;   // Set DCO  = 8MHz
                    DCOCTL = CALDCO_8MHZ;                     
                    P4OUT |=BIT0;           // Microphone ON
                    P2OUT |=BIT4;           // Offset Voltage ON     
                    OA0CTL0 |=OAPM_3;       // Fast mode for Gain Op-amp                     
                    OA1_delay_ON = 1;       // Set OA1_delay _ON flag
                    OA1CTL0 |=OAPM_3;       // Fast mode for filter Op-amp
                                            
                                            // Configure Timer_A to provide a delay 
                                            // of 3.2ms for filter to settle
                    TACCR0 = 32;            
                    TACCTL0 = CCIE;
                    TACTL |= MC_1;          // Up mode, TA start      
                     
                    __bis_SR_register(LPM3_bits + GIE); // LPM3, enable interrupts
                    
                    sum = 0;                
                    sample_count = 0;
                    activity =1;            // Set activity flag
                     
                    ADC10CTL0 &= ~ENC;      // Disable ADC and rec-configure
                    ADC10CTL0 = ADC10SHT_1+ADC10SR;   // Sample and hold time = 8xADC10CLKS   
                    ADC10CTL1= INCH_13+ADC10DIV_2+ADC10SSEL_3+CONSEQ_0; // input channel A13,   
                                            // single-channel-single-conversion mode   
                                                              
                                            // Configure Timer_B to interrupt every 0.25ms 
                    TBCTL= TBSSEL_2;        // TBCLK = SMCLK = 8MHz        
                    TBCCTL0 = CCIE;         // Capture compare interrupt enable   
                    TBCCR0 = 1999;          // Load TACCR0 with 0.25ms count
                    TBCTL |= MC_1;          // Up mode, TA start                    
                } 
                else                      
                {
                    // No ACTVITY Detected
                    ADC10CTL0 &= ~ENC;      
                    ADC10CTL0 &= ~ADC10ON;  // Power-off ADC10  
                    TACCTL0 = CCIE;         
                    TACCR0 = TACCR0_2ms;    
                    TACTL |= MC_1;          // Re-enable Timer_A for 2.5ms periodic wake-up
                }
            }
    }
}

// Timer_A0 Interrupt Service Routine
#pragma vector=TIMERA0_VECTOR
__interrupt void TIMER_ISR(void)
{
    if(!OA1_delay_ON)                       // Is Filter_Opamp delay ON?
    {
        TACTL &=~(MC_1);                    // Stop Timer_A
    }
    else
    {
        TACTL &=~(MC_1);                    // Stop Timer_A
        OA1_delay_ON=0;                     // Reset the OA1_delay_ON flag
    }
    
    __bic_SR_register_on_exit(LPM3_bits+GIE); // Clear LPM bits upon ISR Exit
}

// Timer_B0 Interrupt Service Routine
#pragma vector=TIMERB0_VECTOR
__interrupt void TIMERB_ISR(void)
{
       ADC10CTL0 |= ADC10ON;
       ADC10CTL0 |= ENC + ADC10SC;          // Enable ADC10 + start conversion 
       while((ADC10BUSY & ADC10CTL0));      // Check for conversion complete  
       temp1= ADC10MEM-530;                 // Conv result - op-amp ref voltage
       sample_count++;                      // Increment sample_count
       wdf_thud(temp1);                     // Call the LWDF for thud detection
        if (output1 & 0x8000)               // Convert numbers with sign bit = 1 to -ve numbers
            output1 = ~output1 + 1;
        sum=sum+output1;                    // Accumulate LWDF output in variable sum
        if(sample_count==128)               // 128 samples accumulated? (32ms over?)
        {
            TBCTL &=~(MC_1);                // Stop Timer_B               
            sample_count=0;                 // clear sample_count
            average = (sum) >> 7;           // Average = sum/128
            sum = 0;                        // Clear sum
            if (average >= THUD_LVL)        // Compare average with Thud threshold      
            {
                // VALID THUD Detected
                init_glass_break();
            }
            else
            {
                // VALID THUD not confirmed
                BCSCTL1= CALBC1_12MHZ;      // Set DCO = 12MHz
                DCOCTL = CALDCO_12MHZ;                   
                OA0CTL0 &= ~(OAPM_3);       // OA0 Gain-Op-amp OFF        
                OA1CTL0 &= ~(OAPM_3);       // OA1 Filter OFF
                ADC10CTL0 &= ~ENC + ADC10IE;// Power-off ADC10 & "disable interrupts" - not reqd
                ADC10CTL0 &= ~ADC10ON;  
                                            // Re-configure ADC10
                ADC10CTL0 = ADC10SHT_1+ADC10SR;  // Sample and hold time = 8xADC10CLKS                  
                ADC10CTL1= INCH_1+ADC10DIV_2+ADC10SSEL_3+CONSEQ_0;// input channel A1,   
                                            // single-channel-single-conversion mode                    
                P4OUT &= ~BIT0;             // Microphone OFF
                P2OUT &= ~BIT4;             // Offset Voltage OFF     
                INIT_wdf_2k();              // Clear all the WDF variables
                activity=0;                 // Clear activity flag
                TACCTL0 = CCIE; 
                TACCR0 = TACCR0_2ms;
                TACTL |= MC_1;              // Re-enable Timer_A for 2.5ms periodic wake-up
                __bic_SR_register_on_exit(LPM0_bits);     // Clear LPM0 bits on ISR exit
                __bis_SR_register_on_exit(LPM3_bits+GIE); // Enter LPM3 on ISR exit             
            }
        }       
}
   
// ADC10 Interrupt Service Routine 
#pragma vector=ADC10_VECTOR
__interrupt void ADC10_ISR(void)
{
    temp1= ADC10MEM-530;                    // Conv result - op-amp ref voltage
    sample_count++;                         // Increment sample_count
    {
        signal_analysis_one();              
        if (sample_count ==2336)            // 2336 samples accumulated? (60ms over?)
        {
            signal_analysis_two();
                                            // Reconfigure ADC10
            ADC10CTL0 = ADC10SHT_1+ADC10SR; // Sample and hold time = 8xADC10CLKS 
            ADC10CTL1= INCH_1+ADC10DIV_2+ADC10SSEL_3+CONSEQ_0; // input channel A1,   
                                            // single-channel-single-conversion mode                   
            activity=0;                     // clear activity flag
                                            // Re-enable Timer_A for 2.5ms periodic wake-up
            TACCTL0 = CCIE; 
            TACCR0 = TACCR0_2ms;
            TACTL |= MC_1;                  // Up-mode, start Timer_A  
            __bic_SR_register_on_exit(LPM0_bits);     // Clear LPM0 bits on ISR exit
            __bis_SR_register_on_exit(LPM3_bits+GIE); // Enter LPM3 on ISR exit     
        }  
    }
}
void init_glass_break()
{
    BCSCTL1= CALBC1_12MHZ;                  // Set DCO = 12Mhz
    DCOCTL = CALDCO_12MHZ;
    OA1CTL0 &= ~(OAPM_3);                   // Turn off OA1 (2 kHz AAF filter)
                                            // Initialize all variables used in 
    INITMEM();                              // Signal Analysis one and two
    INIT_WDF();
    sample_count = 0;                       // Clear sample_count
    ADC10CTL0 &= ~ENC;                      // Disable ADC10
    
                                            // Configure ADC10 continuous conv at
                                            // ~38.9 ksps sampling rate
    ADC10CTL0= ADC10SHT_3+ADC10SR+MSC+ADC10ON+ADC10IE; // 64 x ADC10CLKs, 
                                            // ref buff sampling rate supports upto ~50ksps,
                                            // multiple sample and conv, interrupt enable
    ADC10CTL1= INCH_1+ADC10DIV_3+ADC10SSEL_3+CONSEQ_2; // A1, ADC10CLK/4, ADC10CLK = SMCLK,
                                            // Repeat-single-channel 
    ADC10CTL0 |= ENC+ADC10SC;               // Start conversion 
}



