/*
 * video.c
 *
 * ============================================================================
 * Copyright (c) Texas Instruments Inc 2005
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied or provided.
 * ============================================================================
 */

/* Standard Linux headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sched.h>
#include <pthread.h>
#include <asm/types.h>
#include <sys/resource.h>
#include <linux/videodev2.h>
#include <timerutil.h>

/* Codec Engine headers */
#include <xdc/std.h>
#include <ti/sdo/ce/Engine.h>
#include <ti/sdo/ce/osal/Memory.h>
#include <ti/sdo/ce/video/viddec.h>

/* Demo headers */
#include <rendezvous.h>
#include <fifoutil.h>
#include "decode.h"
#include "video.h"
#include "display.h"
#include "loader.h"

/* The size of the read buffer */
#define READBUFSIZE              3 * 1024 * 1024

/* The levels of initialization */
#define VIDEOFILEINITIALIZED     0x1
#define INFIFOOPENED             0x2
#define OUTFIFOOPENED            0x4
#define DISPLAYTHREADCREATED     0x8
#define ENGINEOPENED             0x10
#define VIDEODECODERCREATED      0x20
#define READBUFFERALLOCATED      0x40
#define DISPLAYBUFFERSALLOCATED  0x80

/* Negative value of outputID means the decoder kept the buffer */
#define BUFFER_KEPT_SEND_NEW_DST -1
#define BUFFER_KEPT_RESUBMIT_DST -2

/* Structure containing statistics about the frames in a clip */
typedef struct FrameStats {
    int framesRejected;
    int iFrames;
    int pFrames;
    int bFrames;
} FrameStats;

/* Video decoder names */
char *videoDecodeAlgNames[NUM_VIDEO_DECODERS] = {
    "mpeg4dec",
    "h264dec",
    "mpeg2dec"
};

/* Local function prototypes */
static int getClipSize(VIDDEC_Handle hDecode, int *widthPtr, int *heightPtr);
static int videoDecodeAlgCreate(Engine_Handle hEngine,
                                VIDDEC_Handle *hDecodePtr,
                                enum VideoDecoder videoDecoder,
				Resolution resolution,
                                int *readSizePtr);
static int decodeVideoBuffer(VIDDEC_Handle hDecode, char *inBuf, int inBufSize,
                             int inputID, char *outBuf, int *bytesProcessed,
                             int *outputIDPtr, FrameStats *stats,
			     Resolution resolution,
                             int *frameDonePtr);

/******************************************************************************
 * getClipSize
 ******************************************************************************/
static int getClipSize(VIDDEC_Handle hDecode, int *widthPtr, int *heightPtr)
{
    VIDDEC_DynamicParams    dynamicParams;
    VIDDEC_Status           decStatus;
    XDAS_Int32              status;

    /* Get the codec status (which includes width and height of last frame) */
    dynamicParams.size = sizeof(VIDDEC_DynamicParams);
    decStatus.size = sizeof(VIDDEC_Status);
    status = VIDDEC_control(hDecode, XDM_GETSTATUS, &dynamicParams,
                            &decStatus);

    if (status != VIDDEC_EOK) {
        ERR("XDM_GETSTATUS failed, status=%ld\n", status);
        return FAILURE;
    }

    /* Set the global variables for OSD update */
    gblSetImageWidth(decStatus.outputWidth);
    gblSetImageHeight(decStatus.outputHeight);

    *widthPtr = decStatus.outputWidth;
    *heightPtr = decStatus.outputHeight;

    return SUCCESS;
}

/******************************************************************************
 * videoDecodeAlgCreate
 ******************************************************************************/
static int videoDecodeAlgCreate(Engine_Handle hEngine,
                                VIDDEC_Handle *hDecodePtr,
                                VideoDecoder videoDecoder,
				Resolution resolution,
                                int *readSizePtr)
{
    VIDDEC_Handle          hDecode;
    VIDDEC_Status          decStatus;
    XDAS_Int32             status;
    VIDDEC_Params          params;
    VIDDEC_DynamicParams   dynamicParams;
    char                  *algName;

    algName = videoDecodeAlgNames[videoDecoder];

    /* Configure Codec Resolution */
    if(resolution == RES_NTSC) {
        params.maxWidth = WIDTH_NTSC;
	params.maxHeight = 480;
    } else if (resolution == RES_PAL) {
        params.maxWidth = WIDTH_PAL;
	params.maxHeight = 576;
    } else if (resolution == RES_VGA) {
        params.maxWidth = WIDTH_NTSC;
	params.maxHeight = 480;
    } else if (resolution == RES_720P) {
        params.maxWidth = WIDTH_720P;
	params.maxHeight = HEIGHT_720P;
    } else if (resolution == RES_1080I) {
        params.maxWidth = WIDTH_1080I;
	params.maxHeight = HEIGHT_1080I;
    }

    params.size              = sizeof(VIDDEC_Params);
    params.maxFrameRate      = 0;
    params.maxBitRate        = 0;
    params.dataEndianness    = XDM_BYTE;
    params.forceChromaFormat = XDM_YUV_422ILE;

    /* Create video decoder instance */
    hDecode = VIDDEC_create(hEngine, algName, &params);
    if (hDecode == NULL) {
        ERR("Failed to open video decode algorithm\n");
        return FAILURE; 
    }

    dynamicParams.size          = sizeof(VIDDEC_DynamicParams);
    dynamicParams.decodeHeader  = XDM_DECODE_AU;
    dynamicParams.displayWidth  = XDM_DEFAULT;
    dynamicParams.frameSkipMode = IVIDEO_NO_SKIP;

    /* Set video decoder dynamic params */
    decStatus.size = sizeof(VIDDEC_Status);
    dynamicParams.size = sizeof(VIDDEC_DynamicParams);
    status = VIDDEC_control(hDecode, XDM_SETPARAMS, &dynamicParams,
                            &decStatus);

    if (status != VIDDEC_EOK) {
        ERR("XDM_SETPARAMS failed, status=%ld\n", status);
        return FAILURE;
    }

    /* Get buffer information from video decoder */
    decStatus.size = sizeof(VIDDEC_Status);
    dynamicParams.size = sizeof(VIDDEC_DynamicParams);
    status = VIDDEC_control(hDecode, XDM_GETBUFINFO, &dynamicParams,
                            &decStatus);

    if (status != VIDDEC_EOK) {
        ERR("XDM_GETBUFINFO failed, status=%ld\n", status);
        return FAILURE;
    }

    *readSizePtr = decStatus.bufInfo.minInBufSize[0];
    *hDecodePtr = hDecode;

    return SUCCESS;
}

/******************************************************************************
 * decodeVideoBuffer
 ******************************************************************************/
static int decodeVideoBuffer(VIDDEC_Handle hDecode, char *inBuf, int inBufSize,
                             int inputID, char *outBuf, int *bytesProcessed,
                             int *outputIDPtr, FrameStats *stats,
			     Resolution resolution,
                             int *frameDonePtr)
{
    VIDDEC_DynamicParams    dynamicParams;
    VIDDEC_InArgs           inArgs;
    VIDDEC_OutArgs          outArgs;
    VIDDEC_Status           decStatus;
    XDM_BufDesc             inBufDesc;
    XDM_BufDesc             outBufDesc;
    XDAS_Int32              inBufSizeArray[1];
    XDAS_Int32              outBufSizeArray[1];
    XDAS_Int32              status;
    TimerUtilObj       timer;
    unsigned long      frameTime;

    dynamicParams.size      = sizeof(VIDDEC_DynamicParams);
    decStatus.size          = sizeof(VIDDEC_Status);

    inBufSizeArray[0]       = inBufSize;

    if(resolution == RES_NTSC) {
      outBufSizeArray[0]      = FRAME_SIZE_NTSC;
      (*frameDonePtr) ^= 1;
    }
    else if(resolution == RES_PAL) {
      outBufSizeArray[0]      = FRAME_SIZE_PAL;
      (*frameDonePtr) ^= 1;
    }
    else if(resolution == RES_VGA) {
      outBufSizeArray[0]      = FRAME_SIZE_NTSC;
      (*frameDonePtr) ^= 1;
    }
    else if(resolution == RES_720P)
      outBufSizeArray[0]      = FRAME_SIZE_720P;
    else if(resolution == RES_1080I) {
      outBufSizeArray[0]      = FRAME_SIZE_1080I;
      //  (*frameDonePtr) ^= 1;
    }

    inBufDesc.bufSizes      = inBufSizeArray;
    inBufDesc.bufs          = (XDAS_Int8 **) &inBuf;
    inBufDesc.numBufs       = 1;

    outBufDesc.bufSizes     = outBufSizeArray;
    outBufDesc.bufs         = (XDAS_Int8 **) &outBuf;
    outBufDesc.numBufs      = 1;

    inArgs.size             = sizeof(VIDDEC_InArgs);
    inArgs.numBytes         = inBufSize;
    inArgs.inputID          = inputID;

    outArgs.size            = sizeof(VIDDEC_OutArgs);

    TimerUtil_reset(&timer);

    /* Decode video buffer */
    status = VIDDEC_process(hDecode, &inBufDesc, &outBufDesc, &inArgs,
                            &outArgs);

    TimerUtil_total(&timer, &frameTime);
    DBG("Decode Frame Time = %lu us\n", frameTime);

    *bytesProcessed = outArgs.bytesConsumed; 

    if (status != VIDDEC_EOK) {
        if (status == VIDDEC_ERUNTIME ||
            outArgs.bytesConsumed == 0 ||
            XDM_ISFATALERROR(outArgs.extendedError)) {

            ERR("VIDDEC_process() failed with a fatal error (%ld ext: %#lx)\n",
                status, outArgs.extendedError);
            return FAILURE;
        }
        else {
            stats->framesRejected++;
            return SUCCESS;
        }
    }

    /* Get frame status to see if this was an interlaced frame */
        status = VIDDEC_control(hDecode, XDM_GETSTATUS, &dynamicParams,
                            &decStatus);
    
    if (status != VIDDEC_EOK) {
        ERR("XDM_GETSTATUS failed, status=%ld\n", status);
        return FAILURE;
    }
    
    if (decStatus.contentType == IVIDEO_INTERLACED) {
      (*frameDonePtr) ^= 1;
      }

    switch (outArgs.decodedFrameType) {
        case IVIDEO_I_FRAME:
            stats->iFrames++;
            break;
        case IVIDEO_P_FRAME:
            stats->pFrames++;
            break;
        case IVIDEO_B_FRAME:
            stats->bFrames++;
            break;
    }

    /* Did the decoder keep the frame? */
    if (outArgs.displayBufs.bufs[0] == NULL) {
      if ((outArgs.decodedFrameType != IVIDEO_B_FRAME) & *frameDonePtr) {
	*outputIDPtr = BUFFER_KEPT_SEND_NEW_DST;
      }
      else {
	*outputIDPtr = BUFFER_KEPT_RESUBMIT_DST;
      }
    }
    else {
      *outputIDPtr = outArgs.outputID;
    }

    return SUCCESS;
}

/******************************************************************************
 * videoThrFxn
 ******************************************************************************/
void *videoThrFxn(void *arg)
{
    BufferElement      flush          = { DISPLAY_FLUSH };
    FrameStats         frameStats     = { 0, 0, 0, 0 };
    Engine_Handle      hEngine        = NULL;
    unsigned int       initMask       = 0;
    int                numDisplayBufs = 0;
    VideoEnv          *envp           = (VideoEnv *) arg;
    void              *status         = THREAD_SUCCESS;
    struct sched_param schedParam;
    pthread_t          displayThread;
    pthread_attr_t     attr;
    VIDDEC_Handle      hDecode;
    DisplayEnv         displayEnv;
    LoaderState        lState;
    int                frameDone;
    int                outputID;
    char              *framePtr;
    int                clipWidth;
    int                clipHeight;
    int                frameSize;
    void              *ret;
    BufferElement      e;
    int                i;
    TimerUtilObj       timer;
    unsigned long      frameTime;

    lState.loop = envp->loop;
    lState.readBufSize = READBUFSIZE;
    lState.doneMask = VIDEO_DONE;

    /* Open the input video file */
    lState.inputFd = open(envp->videoFile, O_RDONLY);

    if (lState.inputFd == -1) {
        ERR("Failed to open %s (%s)\n", envp->videoFile, strerror(errno));
        cleanup(THREAD_FAILURE);
    }

    DBG("Video file successfully opened\n");

    initMask |= VIDEOFILEINITIALIZED;

    /* Open the input fifo */
    if (FifoUtil_open(&displayEnv.inFifo,
                      sizeof(BufferElement)) == FIFOUTIL_FAILURE) {
        ERR("Failed to open input fifo\n");
        cleanup(THREAD_FAILURE);
    }

    initMask |= INFIFOOPENED;

    /* Open the output fifo */
    if (FifoUtil_open(&displayEnv.outFifo,
                      sizeof(BufferElement)) == FIFOUTIL_FAILURE) {
        ERR("Failed to open output fifo\n");
        cleanup(THREAD_FAILURE);
    }

    initMask |= OUTFIFOOPENED;

    /* Initialize the priming synchronization mutex */
    pthread_mutex_init(&displayEnv.prime, NULL);

    /* Initialize the thread attributes */
    if (pthread_attr_init(&attr)) {
        ERR("Failed to initialize thread attrs\n");
        cleanup(THREAD_FAILURE);
    }

    /* Force the thread to use custom scheduling attributes */
    if (pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED)) {
        ERR("Failed to set schedule inheritance attribute\n");
        cleanup(THREAD_FAILURE);
    }

    /* Set the thread to be fifo real time scheduled */
    if (pthread_attr_setschedpolicy(&attr, SCHED_FIFO)) {
        ERR("Failed to set FIFO scheduling policy\n");
        cleanup(THREAD_FAILURE);
    }

    /* Set the thread priority */
    schedParam.sched_priority = sched_get_priority_max(SCHED_FIFO)-2;
    if (pthread_attr_setschedparam(&attr, &schedParam)) {
        ERR("Failed to set scheduler parameters\n");
        cleanup(THREAD_FAILURE);
    }

    /* Create the thread */
    displayEnv.hRendezvous = envp->hRendezvous;
    displayEnv.resolution = envp->resolution;
    displayEnv.videoEnv = envp;

    if (pthread_create(&displayThread, &attr, displayThrFxn, &displayEnv)) {
        ERR("Failed to create display thread\n");
        cleanup(THREAD_FAILURE);
    }

    initMask |= DISPLAYTHREADCREATED;

    DBG("Display thread created\n");

    /* Reset, load, and start DSP Engine */
    if ((hEngine = Engine_open(ENGINE_NAME, NULL, NULL)) == NULL) {
        ERR("Failed to open codec engine %s\n", ENGINE_NAME);
        cleanup(THREAD_FAILURE);
    }

    DBG("Codec Engine opened in video thread\n");

    initMask |= ENGINEOPENED;

    /* Allocate and initialize video decoder on the engine */
    if (videoDecodeAlgCreate(hEngine, &hDecode,
                             envp->videoDecoder,
			     envp->resolution,
			     &lState.readSize) == FAILURE) {
        cleanup(THREAD_FAILURE);
    }

    DBG("Video decoder created\n");

    initMask |= VIDEODECODERCREATED;

    lState.readBuffer = (char *) Memory_contigAlloc(lState.readBufSize,
                                                   Memory_DEFAULTALIGNMENT);

    if (lState.readBuffer == NULL) {
        ERR("Failed to allocate contiguous memory block.\n");
        cleanup(THREAD_FAILURE);
    }

    DBG("Contiguous buffer allocated at physical address 0x%lx\n",
        Memory_getPhysicalAddress(lState.readBuffer));

    initMask |= READBUFFERALLOCATED;

    initMask |= DISPLAYBUFFERSALLOCATED;

    /* Signal that initialization is done and wait for other threads */
    Rendezvous_meet(envp->hRendezvous);

prime:
    /* Lock the display priming mutex */
    pthread_mutex_lock(&displayEnv.prime);

    /* Tell the display thread that we are priming */
    e.id = DISPLAY_PRIME;
    if (FifoUtil_put(&displayEnv.outFifo, &e) == FIFOUTIL_FAILURE) {
        ERR("Failed to put buffer to output fifo\n");
        cleanup(THREAD_FAILURE);
    }

    /* Prime the file loader */
    if (loaderPrime(&lState, &framePtr) == FAILURE) {
        cleanup(THREAD_FAILURE);
    }

    outputID = 0;
    frameDone = 1;

    /* Prime the display thread with buffers */
    for (i=0; i < DISPLAY_BUFFERS; i++) {
        /* If interlaced frame, use same destination for both fields */
        if (outputID == BUFFER_KEPT_RESUBMIT_DST) {
            i--;
        }

        if (decodeVideoBuffer(hDecode, framePtr, lState.readSize, i,
                              envp->bufferElements[i].frameBuffer, &frameSize,
                              &outputID, &frameStats, 
			      envp->resolution, &frameDone) == FAILURE) {
            cleanup(THREAD_FAILURE);
        }

        if (lState.firstFrame) {
            if (getClipSize(hDecode, &clipWidth, &clipHeight) == FAILURE) {
                cleanup(THREAD_FAILURE);
            }

	    switch(envp->resolution) {
	    case RES_NTSC:
	      if((clipWidth != WIDTH_NTSC) || (clipHeight != 480)) {
		ERR("Video File is not NTSC resolution.  It is %d x %d.\n",
		    clipWidth, clipHeight);
		cleanup(THREAD_FAILURE);
	      }
	      break;
	    case RES_PAL:
	      if((clipWidth != WIDTH_PAL) || (clipHeight != 576)) {
		ERR("Video File is not PAL resolution.  It is %d x %d.\n",
		    clipWidth, clipHeight);
		cleanup(THREAD_FAILURE);
	      }
	      break;
	    case RES_VGA:
	      if(((clipWidth != WIDTH_VGA) && (clipWidth != WIDTH_NTSC)) || 
		  (clipHeight != 480)) {
		ERR("Video File is not VGA resolution.  It is %d x %d.\n",
		    clipWidth, clipHeight);
		cleanup(THREAD_FAILURE);
	      }
	      break;
	    case RES_720P:
	      if((clipWidth != WIDTH_720P) || (clipHeight != HEIGHT_720P)) {
		ERR("Video File is not 720P resolution.  It is %d x %d.\n",
		    clipWidth, clipHeight);
		cleanup(THREAD_FAILURE);
	      }
	      break;
	    case RES_1080I:
	      if((clipWidth != WIDTH_1080I) || 
		 ((clipHeight != HEIGHT_1080I) && (clipHeight != 1080))) {
		ERR("Video File is not 1080i resolution.  It is %d x %d.\n",
		    clipWidth, clipHeight);
		cleanup(THREAD_FAILURE);
	      }
	      break;
	    }
        }

        /* Read a new frame of encoded data from disk */
        if (loaderGetFrame(&lState, frameSize, &framePtr) == FAILURE) {
            cleanup(THREAD_FAILURE);
        }

        if (lState.endClip) {
            ERR("Clip needs to have at least %d frames\n", DISPLAY_BUFFERS);
            cleanup(THREAD_FAILURE);
        }

        if (outputID >= 0) {
            envp->bufferElements[outputID].width  = clipWidth;
            envp->bufferElements[outputID].height = clipHeight;

            if (FifoUtil_put(&displayEnv.outFifo,
                             &envp->bufferElements[outputID]) == FIFOUTIL_FAILURE) {
                ERR("Failed to put buffer in output fifo\n");
                cleanup(THREAD_FAILURE);
            }  

            numDisplayBufs++;
        }
    }

    /* Release the display thread */
    pthread_mutex_unlock(&displayEnv.prime);

    DBG("Entering video main loop.\n");
    while (!gblGetQuit()) {
        /* Get a used buffer from display thread if required */
        if (outputID != BUFFER_KEPT_RESUBMIT_DST) {
            /* Receive a buffer with a displayed frame from display thread */
            if (FifoUtil_get(&displayEnv.inFifo, &e) == FIFOUTIL_FAILURE) {
                ERR("Failed to get buffer from input fifo\n");
                breakLoop(THREAD_FAILURE);
            }

            /* Is the display thread flushing the pipe? */
            if (e.id == DISPLAY_FLUSH) {
                breakLoop(THREAD_SUCCESS);
            }

            numDisplayBufs--;
        }

	// NF: If you uncomment the TimerUtil functions surrounding
	// the decoder, you can see a print out of the actual decode
	// time in microseconds on your console window.  This can be
	// used for performance benchmarks for 720p and 1080i modes.
	
	//	 TimerUtil_reset(&timer);

        /* Decode the encoded frame to the display frame buffer */
        if (decodeVideoBuffer(hDecode, framePtr, lState.readSize, e.id,
                              e.frameBuffer, &frameSize, &outputID,
                              &frameStats, envp->resolution,
			      &frameDone) == FAILURE) {
            breakLoop(THREAD_FAILURE);
        }

	//TimerUtil_total(&timer, &frameTime);
	//DBG("Decode Frame Time = %lu us\n", frameTime);

        /* If the codec didn't keep the buffer, send it to the display thread */
        if (outputID >= 0) {
            envp->bufferElements[outputID].width  = clipWidth;
            envp->bufferElements[outputID].height = clipHeight;

            if (FifoUtil_put(&displayEnv.outFifo,
                             &envp->bufferElements[outputID]) == FIFOUTIL_FAILURE) {
                ERR("Failed to put buffer to output fifo\n");
                breakLoop(THREAD_FAILURE);
            }

            numDisplayBufs++;
        }

	//		 TimerUtil_reset(&timer);

        /* Read a new frame of encoded data from disk */
        if (loaderGetFrame(&lState, frameSize, &framePtr) == FAILURE) {
            breakLoop(THREAD_FAILURE);
        }

	//TimerUtil_total(&timer, &frameTime);
	//DBG("Frame Read Time = %lu us\n", frameTime);


        if (lState.endClip) {
            /* Drain the display thread */
            while (numDisplayBufs > 0) {
                if (FifoUtil_get(&displayEnv.inFifo, &e) == FIFOUTIL_FAILURE) {
                    ERR("Failed to get buffer from input fifo\n");
                    breakLoop(THREAD_FAILURE);
                }

                /* Is the display thread flushing the pipe? */
                if (e.id == DISPLAY_FLUSH) {
                    breakLoop(THREAD_SUCCESS);
                }

                numDisplayBufs--;
            }

            /* Recreate the algorithm */
            VIDDEC_delete(hDecode);
            if (videoDecodeAlgCreate(hEngine, &hDecode, envp->videoDecoder,
				     envp->resolution,
                                     &lState.readSize) == FAILURE) {
                breakLoop(THREAD_FAILURE);
            }

            /* Reprime the file loader and display thread */
            goto prime;
        }

        gblIncVideoBytesEncoded(frameSize);
    }

    printf("\nTotal I-frames: %d, P-frames: %d, B-frames: %d, rejected: %d\n",
           frameStats.iFrames, frameStats.pFrames, frameStats.bFrames,
           frameStats.framesRejected);

cleanup:
    /* Make sure the other threads aren't waiting for init to complete */
    Rendezvous_force(envp->hRendezvous);

    /* Make sure the video thread isn't stuck in FifoUtil_get() */
    FifoUtil_put(&displayEnv.outFifo, &flush);

    /* Drain the display thread */
    while (numDisplayBufs-- > 0 && e.id != DISPLAY_FLUSH) {
        if (FifoUtil_get(&displayEnv.inFifo, &e) == FIFOUTIL_FAILURE) {
            ERR("Failed to get buffer from input fifo\n");
            break;
        }
    }

    if (initMask & READBUFFERALLOCATED) {
        Memory_contigFree(lState.readBuffer, lState.readBufSize);
    }

    if (initMask & VIDEODECODERCREATED) {
        VIDDEC_delete(hDecode);
    }

    if (initMask & ENGINEOPENED) {
        Engine_close(hEngine);
    }

    if (initMask & DISPLAYTHREADCREATED) {
        if (pthread_join(displayThread, &ret) == 0) {
            status = ret;
        }
        pthread_mutex_destroy(&displayEnv.prime);
    }

    if (initMask & OUTFIFOOPENED) {
        FifoUtil_close(&displayEnv.outFifo);
    }

    if (initMask & INFIFOOPENED) {
        FifoUtil_close(&displayEnv.inFifo);
    }

    if (initMask & VIDEOFILEINITIALIZED) {
        close(lState.inputFd);
    }

    return status;
}

