/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
//         DualRaySmokeAFE_App_GUIComm.c
//
// Implementation of GUI communication for Dual Ray Smoke Application
// Texas Instruments, Inc.
// Ver 1.0. July 2019
// Lixin.C
// ******************************************************************************
#include <stdlib.h>
#include <string.h>
#include <IQmathLib.h>
#include <DualRaySmokeAFELib_Config.h>
#include <DualRaySmokeAFE_App.h>
#ifdef __ENABLE_GUI__
#include <DualRaySmokeAFE_GUI.h>
#endif
#include "DualRaySmokeAFE_App_Config.h"

static bool getSwitchValuefromString(char * string);

static void DualRaySmokeAFE_App_GUIComm_SendAppConfig(char * string);
static void DualRaySmokeAFE_App_GUIComm_RestoreDefaultAppConfig(char * string);
static void DualRaySmokeAFE_App_GUIComm_SendAlarmMode(void);
static void DualRaySmokeAFE_App_GUIComm_SendHornMode(void);
static void DualRaySmokeAFE_App_GUIComm_SendTempSensorMode(void);
static void DualRaySmokeAFE_App_GUIComm_SendRTCCalibEn(void);
static void DualRaySmokeAFE_App_GUIComm_SendSamplingPeriod(void);
static void DualRaySmokeAFE_App_GUIComm_SendBlueLEDStrength(void);
static void DualRaySmokeAFE_App_GUIComm_SendRedREDStrength(void);
static void DualRaySmokeAFE_App_GUIComm_SendPGAGainRed(void);
static void DualRaySmokeAFE_App_GUIComm_SendPGAGainBlue(void);
static void DualRaySmokeAFE_App_GUIComm_SendTIABiasVoltageRed(void);
static void DualRaySmokeAFE_App_GUIComm_SendTIABiasVoltageBlue(void);
static void DualRaySmokeAFE_App_GUIComm_SendPGABiasVoltageRed(void);
static void DualRaySmokeAFE_App_GUIComm_SendPGABiasVoltageBlue(void);
static void DualRaySmokeAFE_App_GUIComm_SendBlueThreshold(void);
static void DualRaySmokeAFE_App_GUIComm_SendRedThreshold(void);

static void GUICmdUpdateSamplingPeriod(char * string);
static void GUICmdUpdateRTCCalib(char * string);
static void GUICmdUpdateTempSensorEnable(char * string);
static void GUICmdUpdateAlarmEn(char * string);
static void GUICmdUpdateHornEn(char * string);
static void GUICmdUpdateBlueLEDStrength(char * string);
static void GUICmdUpdateRedLEDStrength(char * string);
static void GUICmdUpdatePGAGainRed(char * string);
static void GUICmdUpdatePGAGainBlue(char * string);
static void GUICmdUpdateTIABiasVoltageRed(char * string);
static void GUICmdUpdateTIABiasVoltageBlue(char * string);
static void GUICmdUpdatePGABiasVoltageRed(char * string);
static void GUICmdUpdatePGABiasVoltageBlue(char * string);
static void GUICmdUpdateBlueThreshold(char * string);
static void GUICmdUpdateRedThreshold(char * string);


#ifdef __ENABLE_GUI__
//! \brief RX Command structure.
//!         The corresponding callback will be called when the command is
//!         received from GUI.
static const tDualRaySmokeAFE_GUI_RxCmd GUI_RXCommands[] =
{
     {"bReadConfig",    DualRaySmokeAFE_App_GUIComm_SendAppConfig},
     {"bRestoreConfig", DualRaySmokeAFE_App_GUIComm_RestoreDefaultAppConfig},
     {"tSample",        GUICmdUpdateSamplingPeriod},
     {"RTCCalibEn",     GUICmdUpdateRTCCalib},
     {"tempSensorEn",   GUICmdUpdateTempSensorEnable},
     {"alarmEn",        GUICmdUpdateAlarmEn},
     {"hornEn",         GUICmdUpdateHornEn},
     {"bLS",            GUICmdUpdateBlueLEDStrength},
     {"irLS",           GUICmdUpdateRedLEDStrength},
     {"bPGAGain",       GUICmdUpdatePGAGainBlue},
     {"irPGAGain",      GUICmdUpdatePGAGainRed},
     {"irtiaBias",      GUICmdUpdateTIABiasVoltageRed},
     {"btiaBias",       GUICmdUpdateTIABiasVoltageBlue},
     {"irpgaBias",      GUICmdUpdatePGABiasVoltageRed},
     {"bpgaBias",       GUICmdUpdatePGABiasVoltageBlue},
     {"bLEDTh",         GUICmdUpdateBlueThreshold},
     {"irLEDTh",        GUICmdUpdateRedThreshold},
};

//! \brief TX Command structure.
//!         The device will all commands and corresponding formatted data
//!         when DualRaySmokeAFE_App_GUIComm_SendAppData is called.
static const tDualRaySmokeAFE_GUI_TxCmd GUI_TXCmdData[] =
{
     { "{\"dR\":%d}\n",
          (void *) &sDualRaySmokeAFELib_Data.sLED[DUALRAYSMOKEAFE_LED_RED].u16Dark     },
     { "{\"lR\":%d}\n",
          (void *) &(sDualRaySmokeAFELib_Data.sLED[DUALRAYSMOKEAFE_LED_RED].u16Light)  },
     { "{\"fR\":%d}\n",
          (void *) &(sDualRaySmokeAFELib_Data.sLED[DUALRAYSMOKEAFE_LED_RED].i16Diff)   },
     { "{\"dB\":%d}\n",
          (void *) &(sDualRaySmokeAFELib_Data.sLED[DUALRAYSMOKEAFE_LED_BLUE].u16Dark)  },
     { "{\"lB\":%d}\n",
          (void *) &(sDualRaySmokeAFELib_Data.sLED[DUALRAYSMOKEAFE_LED_BLUE].u16Light) },
     { "{\"fB\":%d}\n",
          (void *) &(sDualRaySmokeAFELib_Data.sLED[DUALRAYSMOKEAFE_LED_BLUE].i16Diff)  },
};

void DualRaySmokeAFE_App_GUIComm_Init(void)
{
    DualRaySmokeAFE_GUI_Init(&GUI_RXCommands[0] ,
                          sizeof(GUI_RXCommands)/sizeof(GUI_RXCommands[0]));
}

void DualRaySmokeAFE_App_GUIComm_SendAppData(void)
{
    DualRaySmokeAFE_GUI_TransmitData(&GUI_TXCmdData[0],
                                  sizeof (GUI_TXCmdData) /
                                  sizeof (GUI_TXCmdData[0]));

}

void DualRaySmokeAFE_App_GUIComm_SendTemperature(void)
{
    char string[20];
    sprintf(string,
            "{\"tempC\":%.2f}\n",
            _IQ15toF(sDualRaySmokeAFELib_Data.iq15TemperatureC));
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

void DualRaySmokeAFE_App_GUIComm_SendConfig(void)
{
    // Send the configuration
    DualRaySmokeAFE_App_GUIComm_SendAlarmStatus();
    DualRaySmokeAFE_App_GUIComm_SendAlarmMode();
    DualRaySmokeAFE_App_GUIComm_SendHornMode();
    DualRaySmokeAFE_App_GUIComm_SendTempSensorMode();
    DualRaySmokeAFE_App_GUIComm_SendRTCCalibEn();
    DualRaySmokeAFE_App_GUIComm_SendSamplingPeriod();
    DualRaySmokeAFE_App_GUIComm_SendBlueLEDStrength();
    DualRaySmokeAFE_App_GUIComm_SendRedREDStrength();
    DualRaySmokeAFE_App_GUIComm_SendPGAGainRed();
    DualRaySmokeAFE_App_GUIComm_SendPGAGainBlue();
    DualRaySmokeAFE_App_GUIComm_SendTIABiasVoltageRed();
    DualRaySmokeAFE_App_GUIComm_SendTIABiasVoltageBlue();
    DualRaySmokeAFE_App_GUIComm_SendPGABiasVoltageRed();
    DualRaySmokeAFE_App_GUIComm_SendPGABiasVoltageBlue();
    DualRaySmokeAFE_App_GUIComm_SendBlueThreshold();
    DualRaySmokeAFE_App_GUIComm_SendRedThreshold();
}

void DualRaySmokeAFE_App_GUIComm_SendAlarmStatus(void)
{

    if (sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmOn == true)
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"aS\":true}\n");
    }
    else
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"aS\":false}\n");
    }
}

//  ***** Local Functions to Send Data to GUI *****     //

static void DualRaySmokeAFE_App_GUIComm_SendAppConfig(char * string)
{
    // Set flag so that configuration is sent in main loop
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void DualRaySmokeAFE_App_GUIComm_RestoreDefaultAppConfig(char * string)
{
    sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmOn =
            DUALRAYSMOKEAFE_APP_DEFAULT_ALARMON;
    sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmDetectionEnabled =
            DUALRAYSMOKEAFE_APP_DEFAULT_ALARMDETECTIONENABLED;
    sDualRaySmokeAFE_AppConfig.APPSTATUS.HornEnabled =
            DUALRAYSMOKEAFE_APP_DEFAULT_HORNENABLED;
    sDualRaySmokeAFE_AppConfig.APPSTATUS.TemperatureMeasEnabled =
            DUALRAYSMOKEAFE_APP_DEFAULT_TEMPERATUREMEASENABLED;
    sDualRaySmokeAFE_AppConfig.APPSTATUS.RTCCalibEnabled =
            DUALRAYSMOKEAFE_APP_DEFAULT_RTCCALIBENABLED;
    sDualRaySmokeAFE_AppConfig.u16MeasIntervalms =
            DUALRAYSMOKEAFE_APP_DEFAULT_ULPTIMER_INTERVAL_MSEC;
    sDualRaySmokeAFE_AppConfig.u16MeasIntervalms_tempor =
            DUALRAYSMOKEAFE_APP_DEFAULT_ULPTIMER_INTERVAL_MSEC;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16LED_CurrentmA[DUALRAYSMOKEAFE_LED_BLUE] =
            DUALRAYSMOKEAFELIB_DEFAULT_BLUE_CURRENT_MA;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16LED_CurrentmA[DUALRAYSMOKEAFE_LED_RED] =
            DUALRAYSMOKEAFELIB_DEFAULT_IR_CURRENT_MA;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGA_Gain[DUALRAYSMOKEAFE_LED_RED] =
            DUALRAYSMOKEAFELIB_DEFAULT_IR_PGAGAIN;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGA_Gain[DUALRAYSMOKEAFE_LED_BLUE] =
            DUALRAYSMOKEAFELIB_DEFAULT_BLUE_PGAGAIN;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16TIABiasmV[DUALRAYSMOKEAFE_LED_RED] =
            DUALRAYSMOKEAFELIB_DEFAULT_IR_TIABIASMV;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16TIABiasmV[DUALRAYSMOKEAFE_LED_BLUE] =
            DUALRAYSMOKEAFELIB_DEFAULT_BLUE_TIABIASMV;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGABiasmV[DUALRAYSMOKEAFE_LED_RED] =
            DUALRAYSMOKEAFELIB_DEFAULT_IR_PGABIASMV;
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGABiasmV[DUALRAYSMOKEAFE_LED_BLUE] =
            DUALRAYSMOKEAFELIB_DEFAULT_BLUE_PGABIASMV;
    sDualRaySmokeAFELib_Config.i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_BLUE] =
            DUALRAYSMOKEAFELIB_DEFAULT_BLUE_THRESHOLD;
    sDualRaySmokeAFELib_Config.i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_RED] =
            DUALRAYSMOKEAFELIB_DEFAULT_IR_THRESHOLD;

    // Update Configuration
    DualRaySmokeAFE_App_Config_SetMeasIntervalmsec(
                    DUALRAYSMOKEAFE_APP_DEFAULT_ULPTIMER_INTERVAL_MSEC);
    // Set flag so that configuration is updated and sent to GUI in main loop
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void DualRaySmokeAFE_App_GUIComm_SendAlarmMode(void)
{

    if (sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmDetectionEnabled == true)
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"alarmEn\":true}\n");
    }
    else
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"alarmEn\":false}\n");
    }
}

static void DualRaySmokeAFE_App_GUIComm_SendHornMode(void)
{

    if (sDualRaySmokeAFE_AppConfig.APPSTATUS.HornEnabled == true)
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"hornEn\":true}\n");
    }
    else
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"hornEn\":false}\n");
    }
}

static void DualRaySmokeAFE_App_GUIComm_SendTempSensorMode(void)
{

    if (sDualRaySmokeAFE_AppConfig.APPSTATUS.TemperatureMeasEnabled == true)
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"tempSensorEn\":true}\n");
    }
    else
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"tempSensorEn\":false}\n");
    }
}

static void DualRaySmokeAFE_App_GUIComm_SendRTCCalibEn(void)
{

    if (sDualRaySmokeAFE_AppConfig.APPSTATUS.RTCCalibEnabled != 0)
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"RTCCalibEn\":true}\n");
    }
    else
    {
        DualRaySmokeAFE_GUI_TransmitStringBlocking("{\"RTCCalibEn\":false}\n");
    }
}

static void DualRaySmokeAFE_App_GUIComm_SendSamplingPeriod(void)
{
    char string[20];

    sprintf(string,
            "{\"tSample\":%d}\n",
            sDualRaySmokeAFE_AppConfig.u16MeasIntervalms);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendBlueLEDStrength(void)
{
    char string[20];

    sprintf(string,
            "{\"bLS\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16LED_CurrentmA[DUALRAYSMOKEAFE_LED_BLUE]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendRedREDStrength(void)
{
    char string[20];

    sprintf(string,
            "{\"irLS\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16LED_CurrentmA[DUALRAYSMOKEAFE_LED_RED]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendPGAGainRed(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"irPGAGain\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGA_Gain[DUALRAYSMOKEAFE_LED_RED]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendPGAGainBlue(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"bPGAGain\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGA_Gain[DUALRAYSMOKEAFE_LED_BLUE]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendTIABiasVoltageRed(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"irtiaBias\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16TIABiasmV[DUALRAYSMOKEAFE_LED_RED]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendTIABiasVoltageBlue(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"btiaBias\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16TIABiasmV[DUALRAYSMOKEAFE_LED_BLUE]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendPGABiasVoltageRed(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"irpgaBias\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGABiasmV[DUALRAYSMOKEAFE_LED_RED]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendPGABiasVoltageBlue(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"bpgaBias\":%d}\n",
            sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGABiasmV[DUALRAYSMOKEAFE_LED_BLUE]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendBlueThreshold(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"bLEDTh\":%d}\n",
            sDualRaySmokeAFELib_Config.i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_BLUE]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}

static void DualRaySmokeAFE_App_GUIComm_SendRedThreshold(void)
{
    char string[20];

    // Only send RED PGA Gain for now
    sprintf(string,
            "{\"irLEDTh\":%d}\n",
            sDualRaySmokeAFELib_Config.i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_RED]);
    DualRaySmokeAFE_GUI_TransmitStringBlocking(string);
}


//  ***** Local Functions to get Data from GUI and update App config *****     //
static void GUICmdUpdateSamplingPeriod(char * string)
{
    uint16_t interval;

    interval = strtol(string, NULL, 10);
    DualRaySmokeAFE_App_Config_SetMeasIntervalmsec(interval);
}

static void GUICmdUpdateRTCCalib(char * string)
{
    bool flag;

    // Currently implemented as boolean switch, interval could be controlled
    // by GUI if needed
    flag = getSwitchValuefromString(string);
    if (flag == false)
    {
        sDualRaySmokeAFE_AppConfig.APPSTATUS.RTCCalibEnabled = false;
    }
    else
    {
        sDualRaySmokeAFE_AppConfig.APPSTATUS.RTCCalibEnabled = true;
    }
}

static void GUICmdUpdateTempSensorEnable(char * string)
{
    bool flag;

    flag = getSwitchValuefromString(string);
    if (flag == false)
    {
        sDualRaySmokeAFE_AppConfig.APPSTATUS.TemperatureMeasEnabled = false;
    }
    else
    {
        sDualRaySmokeAFE_AppConfig.APPSTATUS.TemperatureMeasEnabled = true;
    }
}

static void GUICmdUpdateAlarmEn(char * string)
{
    bool flag;

    flag = getSwitchValuefromString(string);
    // Turn Off Horn if alarm was active
    if ( (flag == false) &&
         (sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmDetectionEnabled == true) &&
         (sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmOn == true) )
    {
        DualRaySmokeAFE_App_TurnAlarmOff();
    }

    sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmDetectionEnabled = flag;
}

static void GUICmdUpdateHornEn(char * string)
{
    bool flag;

    // Turn Off Horn if alarm was active
    flag = getSwitchValuefromString(string);
    if ( (flag == false) &&
         (sDualRaySmokeAFE_AppConfig.APPSTATUS.HornEnabled == true) &&
         (sDualRaySmokeAFE_AppConfig.APPSTATUS.AlarmOn == true) )
    {
        DualRaySmokeAFE_App_TurnAlarmOff();
    }
    sDualRaySmokeAFE_AppConfig.APPSTATUS.HornEnabled = flag;
}

static void GUICmdUpdateBlueLEDStrength(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16LED_CurrentmA[DUALRAYSMOKEAFE_LED_BLUE]
                                                                        = value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdateRedLEDStrength(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16LED_CurrentmA[DUALRAYSMOKEAFE_LED_RED]
                                                                      = value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdatePGAGainRed(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGA_Gain[DUALRAYSMOKEAFE_LED_RED]
                                                                       = value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdatePGAGainBlue(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGA_Gain[DUALRAYSMOKEAFE_LED_BLUE]
                                                                       = value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdateBlueThreshold(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFELib_Config.i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_BLUE] =
            value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdateRedThreshold(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFELib_Config.i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_RED] =
            value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdateTIABiasVoltageRed(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16TIABiasmV[DUALRAYSMOKEAFE_LED_RED] =
                                                       value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdateTIABiasVoltageBlue(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16TIABiasmV[DUALRAYSMOKEAFE_LED_BLUE] =
                                                           value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdatePGABiasVoltageRed(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGABiasmV[DUALRAYSMOKEAFE_LED_RED] =
                                                        value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static void GUICmdUpdatePGABiasVoltageBlue(char * string)
{
    uint16_t value;

    value = strtol(string, NULL, 10);
    sDualRaySmokeAFE_HALConfig.AFEConfig->u16PGABiasmV[DUALRAYSMOKEAFE_LED_BLUE] =
                                                        value;
    sDualRaySmokeAFE_AppConfig.APPCTL.ConfigUpdate = true;
}

static bool getSwitchValuefromString(char * string)
{
    if (strncmp(string, "true", 4) == 0)
    {
        return true;
    }
    else
    {
        return false;
    }
}

#endif // __ENABLE_GUI__

