//------------------------------------------------------------------------------
// MSP430F42x Single Chip Weigh Scale - C Language Version
//
// Description: Program reads out the MSP430 SD16 Sigma-Delta ADC channel 0.
// The differential input voltage is displayed as a signed number on a 7.5 digit
// LCD. The displayed value is calculated using a 2-point calibration mechanism.
// During initial power-up (with INFOA Flash memory erased), two calibration
// data points are obtained (CalMin and CalMax) and stored into Flash memory.
// The range of CalMin to CalMax is projected during measurement mode into a
// display-value from 0 to CAL_MIN_MAX_SPAN. A button connected to P1.0 is
// used to enter low-power mode (short press) and re-calibrate the
// system (long press). An external 32-kHz watch crystal is used for FLL
// stabilization, time interval generation and for driving the LCD.
//
//
//                  MSP430F427
//               +---------------+
//               |               |    +----------------------+
//   IN+     o---|A0+      S0-S23|--->|   SoftBaugh SBLCDA4  |
//   IN-     o---|A0-   COM0-COM3|--->|  7.1 Digit,4-Mux LCD |
//   VBridge o--+|P2.0           |    +----------------------+
//              +|P2.1           |
//           o---|VRef    R03-R33|<---LCD Voltage Ladder Rs
//               |               |
//               |       XIN/XOUT|<---32.768KHz Watch Crystal
//               |           P1.0|<---Button (low-active)
//               +---------------+
//
// Andreas Dannenberg
// Texas Instruments Inc.
// September 2004
// Built with IAR Embedded Workbench Version: 3.20A
//------------------------------------------------------------------------------
#include "msp430x42x.h"

#define CAL_MIN_MAX_SPAN    10000               // Scale value for CalMin/CalMax
                                                // 10,000 Counts=10kg=10,000g

// Circuit related definitions
#define BRIDGE_SUPPLY       (0x03)              // IO pins P2.0/P2.1 for
                                                // pos. bridge rail
#define PUSH_BUTTON         (0x01)              // Button on pin P1.0

// Get a 18-bit wide result from SD16 channel 0 (17 bit + sign bit)
#define GET_RESULT(var)     SD16CCTL0 &= ~SD16LSBACC; \
                            var = (long)(int)SD16MEM0 << 2; \
                            SD16CCTL0 |= SD16LSBACC; \
                            var |= ((int)SD16MEM0 & 0xc0) >> 6

enum
{
  PM_MEASURE,                                   // Program Mode - Normal
  PM_CAL_LO,                                    // Program Mode - Cal Low
  PM_CAL_HI,                                    // Program Mode - Cal High
  PM_PWRDOWN                                    // Program Mode - Powered Down
};

// LCD segment definitions.
#define e       0x40                            //  AAAA
#define g       0x20                            // F    B
#define f       0x10                            // F    B
#define d       0x08                            //  GGGG
#define c       0x04                            // E    C
#define b       0x02                            // E    C
#define a       0x01                            //  DDDD

const char LCD_Tab[] = {
  a + b + c + d + e + f,                        // Displays "0"
  b + c,                                        // Displays "1"
  a + b + d + e + g,                            // Displays "2"
  a + b + c + d + g,                            // Displays "3"
  b + c + f + g,                                // Displays "4"
  a + c + d + f +g,                             // Displays "5"
  a + c + d + e + f + g,                        // Displays "6"
  a + b + c,                                    // Displays "7"
  a + b + c + d + e + f + g,                    // Displays "8"
  a + b + c + d + f + g,                        // Displays "9"
  a + b + c + e + f + g,                        // Displays "A"
  b + c + e + f + g,                            // Displays "H"
  a + d + e + f,                                // Displays "C"
  d + e + f,                                    // Displays "L"
  0x00,                                         // Displays Blank
  0x00                                          // Displays Blank
};

#undef a
#undef b
#undef c
#undef d
#undef e
#undef f
#undef g

// Global vars
static unsigned int VoltageSettleCtr;           // Used as voltage settle ctr.
static unsigned int SD16TempCtr;                // Number of resuts collected
static long SD16Temp;                           // Temp sum register
static long SD16Result;                         // Final averaged result
static long LastADCValue;                       // Last averaged result
static unsigned int ProgramMode;                // Current program mode
static long CalMinTmp;                          // Temp vars to hold
static long CalMaxTmp;                          // calibration values
static char ButtonDownCtr;                      // Keeps track of button press
                                                // duration

#define FLAG_UPDATE_DISPL     0x01              // Bit definitions used for
#define FLAG_BUTTON_DOWN      0x02              // flag register

static char Flags = FLAG_UPDATE_DISPL;          // Flag register

#pragma dataseg = INFOA                         // Info Flash Memory Block A
__no_init static long CalMin;
__no_init static long CalMax;
#pragma dataseg = default

// Function prototypes
void Init_Sys(void);
void StartNextConversion(void);
void StoreCalInFlash(void);
void Disp_Signed_Long(long Value);
void Disp_BCD(unsigned long Value);
//------------------------------------------------------------------------------
void main(void)
{
  Init_Sys();

  if (CalMin == CalMax)                         // Are constants in Flash OK?
  {
    Disp_BCD(0xfcadfd0);                        // Display 'CAL LO'
    ProgramMode = PM_CAL_LO;                    // Enter calibration mode
  }
  else
  {
    ProgramMode = PM_MEASURE;                   // Enter measurement mode
  }

  StartNextConversion();                        // Start conversions
  __bis_SR_register(LPM0_bits + GIE);           // Enter LPM0 w/ ints enabled
}
//------------------------------------------------------------------------------
void Init_Sys(void)
{
  int i;
  char *pLCD = (char *)&LCDM1;

  WDTCTL = WDTPW + WDTHOLD;                     // Stop WDT
  FLL_CTL0 |= XCAP18PF;                         // Set load capacitance for xtal

  for (i = 0; i < 20; i++)                      // Clear LCD memory
    *pLCD++ = 0;

  LCDCTL = LCDSG0_3 + LCD4MUX + LCDON;          // 4mux LCD, segs0-23
  BTCTL = BT_fLCD_DIV128+BTDIV+BT_fCLK2_DIV64;  // 0.5s BT Int, Set LCD freq
  IE2 |= BTIE;                                  // Enable Basic Timer interrupt

  P1OUT = 0xff;                                 // P1OUTs = 1
  P1DIR = 0xff & ~PUSH_BUTTON;                  // All pins but button to output
  P1IES = PUSH_BUTTON;                          // Button int on falling edge
  P1IFG = 0;
  P1IE = PUSH_BUTTON;                           // Enable button interrupt
  P2OUT = 0xff;                                 // P2OUTs = 1
  P2DIR = 0xff;                                 // All pins outputs

  SD16INCTL0 = SD16GAIN_32 + SD16INCH_0;        // 32x gain, channel pair A0
  SD16CCTL0 = SD16DF + SD16IE;                  // Continuous conv., 2s compl.
}
//------------------------------------------------------------------------------
// Programs the calibration constants CalMinTmp and CalMaxTmp into Flash
// info memory segment A using in-system self programming techniques
//------------------------------------------------------------------------------
void StoreCalInFlash(void)
{
  FCTL2 = FWKEY + FSSEL1 + FN1;                 // SMCLK/3 = ~333kHz
  FCTL3 = FWKEY;                                // Clear LOCK
  FCTL1 = FWKEY + ERASE;                        // Enable segment erase
  *(unsigned int *)0x1080 = 0;                  // Dummy write, erase info A
  FCTL1 = FWKEY + WRT;                          // Enable write
  CalMin = CalMinTmp;                           // Program calibration constants
  CalMax = CalMaxTmp;
  FCTL1 = FWKEY;                                // Done. Clear WRT
  FCTL3 = FWKEY + LOCK;                         // Done. Set LOCK
}
//------------------------------------------------------------------------------
// Converts the 32-Bit integer number 'Value' to BCD and outputs it
// on the LCD display
//------------------------------------------------------------------------------
void Disp_Signed_Long(long Value)
{
  unsigned int i;
  unsigned long Output;
  char fNeg = 0;

  if (Value < 0)                                // Test for negative value
  {
    Value = -Value;                             // Negate value
    fNeg = 1;                                   // Set negative flag
  }

  for (i = 32, Output = 0; i; i--)              // BCD Conversion, 32-Bit
  {
    Output = __bcd_add_long(Output, Output);
    if (Value & 0x80000000)
      Output = __bcd_add_long(Output, 1);
    Value <<= 1;
  }

  if (fNeg)                                     // Display neg sign?
    Output |= 0x80000000;                       // Bit 31 indicates neg. number

  Disp_BCD(Output);
}
//------------------------------------------------------------------------------
// Displays the BCD number 'Value' on the LCD display
//------------------------------------------------------------------------------
void Disp_BCD(unsigned long Value)
{
  char *pLCD = (char *)&LCDM6;
  int i;

  for (i = 0; i < 7; i++)                       // Process 7 digits
  {
    *pLCD++ = LCD_Tab[Value & 0x0f];            // Segments to LCD
    Value >>= 4;                                // Process next digit
  }

  if (Value & 0x01)
    LCDM4 |= 0x08;                              // Display "1" (8th digit)
  else
    LCDM4 &= ~0x08;                             // Clear "1" (8th digit)

  if (Value & 0x08)                             // Bit 31 indicates neg. number
    LCDM4 |= 0x40;                              // Display "-"
  else
    LCDM4 &= ~0x40;                             // Clear "-"
}
//------------------------------------------------------------------------------
void StartNextConversion(void)
{
  SD16Temp = 0;
  SD16TempCtr = 0;
  P2OUT |= BRIDGE_SUPPLY;                       // Power-up bridge sensor
  VoltageSettleCtr = 46;                        // Allow voltages to settle
                                                // (46+4)x244us=12ms
  SD16CCTL0 |= SD16SC;                          // Start conversion
}
//------------------------------------------------------------------------------
#pragma vector = SD16_VECTOR
__interrupt void SD16_ISR(void)
{
  long CurrentResult;

  GET_RESULT(CurrentResult);                    // Read SD16 result, clear IFG

  if (VoltageSettleCtr)                         // Wait for voltages to settle
  {
    VoltageSettleCtr--;                         // Decrement counter
    return;                                     // Exit ISR
  }

  SD16Temp += CurrentResult;                    // Sum up results

  if (++SD16TempCtr >= 256)
  {
    SD16Result = SD16Temp >> 8;                 // Div 256
    SD16CCTL0 &= ~SD16SC;                       // Disable conversions
    P2OUT &= ~BRIDGE_SUPPLY;                    // Power down bridge voltage

    if (ProgramMode == PM_MEASURE)
      if (Flags & FLAG_UPDATE_DISPL || LastADCValue != SD16Result)
      {
        Flags &= ~FLAG_UPDATE_DISPL;            // Reset flag
        LastADCValue = SD16Result;              // Store new value

        Disp_Signed_Long(((long)SD16Result - CalMin) * CAL_MIN_MAX_SPAN /
                        (CalMax - CalMin));
      }

    __bis_SR_register_on_exit(LPM3_bits);       // Enter LPM3 on ISR exit
  }
}
//------------------------------------------------------------------------------
#pragma vector = BASICTIMER_VECTOR
__interrupt void BT_ISR(void)
{
  switch (ProgramMode)
  {
    case PM_MEASURE :
      if (Flags & FLAG_BUTTON_DOWN)
        if (P1IN & PUSH_BUTTON)                 // Was button released?
        {                                       // Yes, then power down circuit
          Flags &= ~FLAG_BUTTON_DOWN;
          IE2 &= ~BTIE;                         // Disable Basic Timer interrupt
          P2OUT &= ~BRIDGE_SUPPLY;              // Power down bridge voltage
          LCDCTL &= ~LCDON;                     // Disable LCD
          SD16CCTL0 &= ~SD16SC;                 // Disable conversions
          IFG2 &= ~BTIFG;                       // Clear Basic Timer int flag
          ProgramMode = PM_PWRDOWN;
          __bis_SR_register_on_exit(LPM3_bits); // Enter LPM3 on ISR exit
          break;
        }
        else if (++ButtonDownCtr > 5)           // Button still pressed,
        {                                       // time expired?
          Disp_BCD(0xfcadfd0);                  // Display 'CAL LO'
          ProgramMode = PM_CAL_LO;              // Enter calibration mode
        }
      StartNextConversion();
      __bic_SR_register_on_exit(SCG1 + SCG0);   // Enter LPM0 on exit
      break;
    case PM_CAL_LO :
    case PM_CAL_HI :
      StartNextConversion();
      __bic_SR_register_on_exit(SCG1 + SCG0);   // Enter LPM0 on exit
      break;
  }
}
//------------------------------------------------------------------------------
#pragma vector = PORT1_VECTOR
__interrupt void PORT1_ISR(void)
{
  volatile unsigned int i;                      // 'volatile' to prevent opt.

  switch (ProgramMode)
  {
    case PM_MEASURE :
      Flags |= FLAG_BUTTON_DOWN;
      ButtonDownCtr = 0;
      break;
    case PM_CAL_LO :
      CalMinTmp = SD16Result;                   // Get conversion result
      Disp_BCD(0xfcadfb1);                      // Display 'CAL HI'
      ProgramMode = PM_CAL_HI;                  // Enter calibration mode (high)
      break;
    case PM_CAL_HI :
      CalMaxTmp = SD16Result;                   // Get conversion result
      if (CalMinTmp == CalMaxTmp)               // Are calibr constants OK?
      {
        Disp_BCD(0xfcadfd0);                    // Display 'CAL LO'
        ProgramMode = PM_CAL_LO;                // Enter calibration mode
      }
      else                                      // Calibration OK
      {
        StoreCalInFlash();                      // Yes, program constants and
        ProgramMode = PM_MEASURE;               // Enter normal mode
        Flags |= FLAG_UPDATE_DISPL;             // Request display update
        Flags &= ~FLAG_BUTTON_DOWN;             // Clear button flag
      }
      break;
    case PM_PWRDOWN :
      IE2 |= BTIE;                              // Enable Basic Timer interrupt
      P2OUT |= BRIDGE_SUPPLY;                   // Power-up bridge sensor
      LCDCTL |= LCDON;                          // Enable LCD
      StartNextConversion();                    // Start conversion
      Flags |= FLAG_UPDATE_DISPL;               // Request display update
      ProgramMode = PM_MEASURE;
      __bic_SR_register_on_exit(SCG1 + SCG0);   // Enter LPM0 on exit
      break;
  }

  for (i = 0x7fff; i; i--);                     // Delay for key-debounce
  P1IFG = 0x00;                                 // Clear all P1 int flags
}
